package com.interactivemesh.j3d.sourcecode.capbits;

// All classes listed for documention reasons 
// import javax.media.j3d.*;

import javax.media.j3d.AlternateAppearance;
import javax.media.j3d.AmbientLight;
import javax.media.j3d.Appearance;
import javax.media.j3d.AuralAttributes;
import javax.media.j3d.Background;
import javax.media.j3d.BackgroundSound;
import javax.media.j3d.Behavior;
import javax.media.j3d.BoundingLeaf;
import javax.media.j3d.BranchGroup;
import javax.media.j3d.Clip;
import javax.media.j3d.ColoringAttributes;
import javax.media.j3d.CompressedGeometry;
import javax.media.j3d.ConeSound;
import javax.media.j3d.DepthComponent;
import javax.media.j3d.DirectionalLight;
import javax.media.j3d.ExponentialFog;
import javax.media.j3d.Fog;
import javax.media.j3d.Geometry;
import javax.media.j3d.GeometryArray;
import javax.media.j3d.Group;
import javax.media.j3d.ImageComponent;
import javax.media.j3d.IndexedGeometryArray;
import javax.media.j3d.Light;
import javax.media.j3d.LineAttributes;
import javax.media.j3d.LinearFog;
import javax.media.j3d.Link;
import javax.media.j3d.Material;
import javax.media.j3d.MediaContainer;
import javax.media.j3d.ModelClip;
import javax.media.j3d.Morph;
import javax.media.j3d.NodeComponent;
import javax.media.j3d.OrderedGroup;
import javax.media.j3d.OrientedShape3D;
import javax.media.j3d.PointAttributes;
import javax.media.j3d.PointLight;
import javax.media.j3d.PointSound;
import javax.media.j3d.PolygonAttributes;
import javax.media.j3d.Raster;
import javax.media.j3d.RenderingAttributes;
import javax.media.j3d.SceneGraphObject;
import javax.media.j3d.ShaderAppearance;
import javax.media.j3d.ShaderAttributeObject;
import javax.media.j3d.ShaderAttributeSet;
import javax.media.j3d.ShaderProgram;
import javax.media.j3d.Shape3D;
import javax.media.j3d.SharedGroup;
import javax.media.j3d.Sound;
import javax.media.j3d.Soundscape;
import javax.media.j3d.SpotLight;
import javax.media.j3d.Switch;
import javax.media.j3d.TexCoordGeneration;
import javax.media.j3d.Text3D;
import javax.media.j3d.Texture;
import javax.media.j3d.Texture2D;
import javax.media.j3d.TextureAttributes;
import javax.media.j3d.TextureUnitState;
import javax.media.j3d.TransformGroup;
import javax.media.j3d.TransparencyAttributes;
import javax.media.j3d.ViewPlatform;
import javax.media.j3d.ViewSpecificGroup;

/**
 * CapAndFrqBits
 *
 * Most subclasses of SceneGraphObject have a set of capability bits and 
 * a corresponding set of capabilityIsFrequent bits.
 * 
 * Capability bits control if a particular attribute of a node or node component 
 * is readable or writable within a live or compiled scene graph for performance 
 * reasons.
 * 
 * Only one capability or capabilityIsFrequent bit may be set or cleared 
 * per method invocation (see SceneGraphObject javadoc). 
 * 
 * These bits (integers) cannot be 'ORed' together, because they are numbered 
 * consecutively.
 * 
 * For each concerned class the capability bits can be considered as an unique 
 * bit position within a long number (64 bit). That's the way Java 3D stores them 
 * internally, but these 'longs' are not public. 
 *
 * This source code allows to retrieve these 'longs' respectively to set and clear 
 * the bits in a convenient way. 
 * 
 * Specific 'longs' can be generated with the program CapBitSelector. 
 * The magic number to set or clear all bits for an arbritary scene graph object 
 * is '281474976710651L' or '2**48 - 1 - 2**2' 
 * (bit positions 0 - 47, position 2 isn't used).
 * 
 *
 * Version: 1.0
 * Date: 2008/02/29
 * 
 * Version: 1.1
 * Date: 2009/10/28
 * 
 * Copyright (c) 2008-2009
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 * 
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * but you may not modify, compile, or distribute this 'CapAndFrqBits.java'.
 */

public final class CapAndFrqBits {

    // Java 3D API 1.5
    // Capability bit mask of SceneGraphObjects
    // For maintenance reasons a class' bit mask doesn't include the masks of its superclasses
    // Mask is identical to the long number when all cap bits are set!

    // Node (Leaf)
    private static long node    =   211106232537083L;

    // Groups
    private static long group   =   126976L;

    private static long bg      =   131072L;
    private static long og      =   393216L;
    private static long sg      =   131072L;
    private static long sw      =   393216L;
    private static long tg      =   393216L;
    private static long vsg     =   393216L;

    // Leaves
    private static long aa      =   258048L;
    private static long bgd     =   4190208L;
    private static long beh     =   0L;
    private static long bl      =   12288L;
    private static long cl      =   61440L;

    private static long fog     =   847872L;
    private static long exf     =   196608L;
    private static long lif     =   196608L;

    private static long light   =   805564416L;
    private static long aml     =   0L;
    private static long dil     =   786432L;
    private static long pol     =   3932160L;
    private static long spl     =   264241152L;

    private static long link    =   12288L;
    private static long mcl     =   1044480L;
    private static long mph     =   4190208L;

    private static long s3d     =   1044480L;
    private static long o3d     =   267386880L;

    private static long sound   =   69273527513088L; // V 1.1
    private static long bgs     =   0L;
    private static long pos     =   64424509440L;
    private static long cos     =   1030792151040L;

    private static long ssc     =   61440L;
    private static long vp      =   12288L;	

    // NodeComponents ( NodeComponent = 0L )
    private static long appearance          =	4194303L;
    private static long shaderAppear        =	62914560L;
    
    private static long colAttr             =	15L;
    private static long lineAttr            =	63L;
    private static long material            =	3L;
    private static long pointAttr           =	15L;
    private static long polyAttr            =	255L;
    private static long rendAttr            =	65535L;
    private static long texture             =	32767L;
    private static long texture2D           =	32768L;	    
    private static long texAttr             =	1023L;
    private static long texCoordGene        =	63L;
    private static long texUnitState        =	3L;
    private static long transAttr           =	63L;
    
    private static long aurAttr             =	268435455L;
    private static long depth               =	3L;
    
    private static long imageComp           =	15L;
    
    private static long geometry            =	262144L;
    private static long geometryArray       =	16384511L;
    private static long indGeomArray        =  	50462208L;
    
    private static long compGeom            =  	15L;
    private static long raster              =  	8191L;
    private static long text3d              =  	8191L;
    
    private static long media               =	15L;

    private static long shaderProgram       =	3L;
    private static long shaderAttrSet       =	3L;
    private static long shaderAttrObject    =	3L;
    
    // Min cap bit  0: Node.ENABLE_COLLISION_REPORTING
    // Max cap bit 47: Node.ALLOW_LOCAL_READ;
    private static int bitCount	= 48;
    
    private CapAndFrqBits() {		
    }

    public static long getCapBits(SceneGraphObject sgo) {

        // Bit mask for sgo's type
        long capMask = capMaskOf(sgo);
        // Return value
        long capBits = 0L;

        // sgo has caps
        if (capMask > 0) {
            // Run through all bit positions
            for (int i=0; i < bitCount; i++) {
                // i-th position
                long bit = (1L << i);
                // Position represents a bit for this sgo ?
                if ((capMask & bit) != 0L) {
                    // If bit is set
                    if (sgo.getCapability(i))
                        capBits |= bit;
                }
            }
        }

        return capBits;
    }
    
    public static long getFrqBits(SceneGraphObject sgo) {

        long capMask = capMaskOf(sgo);

        long frqBits = 0L;

        if (capMask > 0) {
            for (int i=0; i < bitCount; i++) {	    		
                long bit = (1L << i);	    		
                if ((capMask & bit) != 0L) {
                    if (sgo.getCapabilityIsFrequent(i))
                        frqBits |= bit;
                }
            }
        }

        return frqBits;
    }
    
    public static void setCapBits(SceneGraphObject sgo, long capBits) {
    	
        // Bit mask for sgo's type
    	long capMask = capMaskOf(sgo);
    	
    	// sgo has no caps
    	if (capMask == 0)
            return;
    	// Must be positive or 0
    	if (capBits < 0)
            return;
    	
        // Run through all bit positions
    	for (int i=0; i < bitCount; i++) {    	    		
            // i-th position
            long bit = (1L << i);    	    		
            // Position represents a bit for this sgo
            if ((capMask & bit) != 0L) {
                // Set bit
                if (((capBits & bit) != 0L))
                    sgo.setCapability(i);
                 // Clear bit
                else
                    sgo.clearCapability(i);    			
            }    		
    	}
    }
    
    public static void setFrqBits(SceneGraphObject sgo, long frqBits) {
    	
    	long capMask = capMaskOf(sgo);
    	
    	if (capMask == 0)
            return;
    	if (frqBits < 0)
            return;
    	
    	for (int i=0; i < bitCount; i++) {   	    		
            long bit = (1L << i);   	    		
            if ((capMask & bit) != 0L) {
                if (((frqBits & bit) != 0L))
                    sgo.setCapabilityIsFrequent(i);
                else
                    sgo.clearCapabilityIsFrequent(i);    			
            }    		
    	}
    }
    
    private static long capMaskOf(SceneGraphObject sgo) {
    	
    	long capMask = 0;
    	
    	if (sgo instanceof NodeComponent) {
    		
            if (sgo instanceof Material)
                capMask |= material;
            else if (sgo instanceof TransparencyAttributes)
                capMask |= transAttr;

            else if (sgo instanceof PolygonAttributes)
                capMask |= polyAttr;
            else if (sgo instanceof LineAttributes)
                capMask |= lineAttr;
            else if (sgo instanceof ColoringAttributes)
                capMask |= colAttr;
            else if (sgo instanceof RenderingAttributes)
                capMask |= rendAttr;
            else if (sgo instanceof PointAttributes)
                capMask |= pointAttr;

            else if (sgo instanceof Geometry) {
                capMask |= geometry;
                if (sgo instanceof GeometryArray) {
                    capMask |= geometryArray;
                    if (sgo instanceof IndexedGeometryArray)
                        capMask |= indGeomArray;
                }
                else if (sgo instanceof CompressedGeometry)
                    capMask |= compGeom;
                else if (sgo instanceof Raster)
                    capMask |= raster;
                else if (sgo instanceof Text3D)
                    capMask |= text3d;
            }

            else if (sgo instanceof Appearance) {
                capMask |= appearance;   		
                if (sgo instanceof ShaderAppearance)
                    capMask |= shaderAppear;
            }

            else if (sgo instanceof Texture) {
                capMask |= texture;
                if (sgo instanceof Texture2D)
                    capMask |= texture2D;
            }

            else if (sgo instanceof TexCoordGeneration)
                capMask |= texCoordGene;
            else if (sgo instanceof TextureAttributes)
                capMask |= texAttr;
            else if (sgo instanceof TextureUnitState)
                capMask |= texUnitState;

            else if (sgo instanceof ImageComponent)
                capMask |= imageComp;

            else if (sgo instanceof ShaderAttributeObject)
                capMask |= shaderAttrObject;
            else if (sgo instanceof ShaderAttributeSet)
                capMask |= shaderAttrSet;
            else if (sgo instanceof ShaderProgram)
                capMask |= shaderProgram;

            else if (sgo instanceof AuralAttributes)
                capMask |= aurAttr;
            else if (sgo instanceof DepthComponent)
                capMask |= depth;
            else if (sgo instanceof MediaContainer)
                capMask |= media;
    	}
    	else if (sgo instanceof Group) {   
    		
            capMask |= node;
            capMask |= group; // Group

            if (sgo instanceof TransformGroup)
                capMask |= tg;
            else if (sgo instanceof BranchGroup)
                capMask |= bg;
            else if (sgo instanceof SharedGroup)
                capMask |= sg;
            else if (sgo instanceof Switch)
                capMask |= sw;
            else if (sgo instanceof ViewSpecificGroup)
                capMask |= vsg;
            else if (sgo instanceof OrderedGroup) // incl. DecalGroup
               capMask |= og;
    	}
    	// Leaves 
    	else {
    		
            capMask |= node;

            if (sgo instanceof Shape3D) {
                capMask |= s3d;
                if (sgo instanceof OrientedShape3D)
                    capMask |= o3d;
            }

            else if (sgo instanceof AlternateAppearance)
                capMask |= aa;
            else if (sgo instanceof Background)
                capMask |= bgd;
            else if (sgo instanceof Behavior) // incl. all Interpolators, Billboard, LOD
                capMask |= beh;
            else if (sgo instanceof BoundingLeaf)
                capMask |= bl;
            else if (sgo instanceof Clip)
                capMask |= cl;

            else if (sgo instanceof Fog) {
                capMask |= fog;
                if (sgo instanceof ExponentialFog)
                    capMask |= exf;
                else if (sgo instanceof LinearFog)
                    capMask |= lif;
            }

            else if (sgo instanceof Light) {
                capMask |= light;
                if (sgo instanceof AmbientLight)
                    capMask |= aml;
                else if (sgo instanceof DirectionalLight)
                    capMask |= dil;
                else if (sgo instanceof PointLight) {
                   capMask |= pol;
                    if (sgo instanceof SpotLight)
                        capMask |= spl;
                }
            }

            else if (sgo instanceof Link)
                capMask |= link;
            else if (sgo instanceof ModelClip)		
                capMask |= mcl;
            else if (sgo instanceof Morph)
                capMask |= mph;

            else if (sgo instanceof Sound) {
                capMask |= sound;
                if (sgo instanceof BackgroundSound)
                    capMask |= bgs;
                else if (sgo instanceof PointSound) {
                    capMask |= pos;
                    if (sgo instanceof ConeSound)
                        capMask |= cos;
                }
            }

            else if (sgo instanceof Soundscape)
                capMask |= ssc;
            else if (sgo instanceof ViewPlatform)
                capMask |= vp;    		
    	}

    	return capMask;
    }
}
