
package com.interactivemesh.j3d.testspace.jfx.charcubesb;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GradientPaint;
import java.awt.Graphics2D;

import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseEvent;

import java.awt.geom.GeneralPath;

import java.awt.image.BufferedImage;

import java.util.Enumeration;
import java.util.LinkedHashMap;

import javax.media.j3d.Alpha;
import javax.media.j3d.Appearance;
import javax.media.j3d.Background;
import javax.media.j3d.Behavior;
import javax.media.j3d.BoundingBox;
import javax.media.j3d.BoundingSphere;
import javax.media.j3d.Bounds;
import javax.media.j3d.BranchGroup;
import javax.media.j3d.Canvas3D;
import javax.media.j3d.DirectionalLight;
import javax.media.j3d.GeometryArray;
import javax.media.j3d.GraphicsConfigTemplate3D;
import javax.media.j3d.Group;
import javax.media.j3d.ImageComponent2D;
import javax.media.j3d.Locale;
import javax.media.j3d.Material;
import javax.media.j3d.OrderedGroup;
import javax.media.j3d.PhysicalBody;
import javax.media.j3d.PhysicalEnvironment;
import javax.media.j3d.PickInfo;
import javax.media.j3d.PolygonAttributes;
import javax.media.j3d.RotationInterpolator;
import javax.media.j3d.Shape3D;
import javax.media.j3d.Switch;
import javax.media.j3d.Transform3D;
import javax.media.j3d.TransformGroup;
import javax.media.j3d.TransparencyAttributes;
import javax.media.j3d.View;
import javax.media.j3d.ViewPlatform;
import javax.media.j3d.VirtualUniverse;
import javax.media.j3d.WakeupCriterion;
import javax.media.j3d.WakeupOnBehaviorPost;

import javax.swing.JPanel;
import javax.swing.SwingUtilities;

import javax.vecmath.AxisAngle4d;
import javax.vecmath.Color3f;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;

import com.sun.j3d.utils.pickfast.PickCanvas;

// FXCanvas3D API 3.0, see http://www.interactivemesh.org/testspace/j3dmeetsjfx.html
import com.interactivemesh.j3d.community.gui.FXCanvas3DSB;
import com.interactivemesh.j3d.community.gui.FXCanvas3DRepainter;

// AWTShapeExtruder API 3.0, see: http://www.interactivemesh.org/testspace/awtshapeextruder.html
import com.interactivemesh.j3d.community.utils.geometry.AWTShapeExtruder;
import com.interactivemesh.j3d.community.utils.geometry.AWTShapeExtrusion;
import com.interactivemesh.j3d.community.utils.geometry.String3D;

// OrbitBehaviorInterim API 2.1, see http://www.interactivemesh.org/testspace/orbitbehavior.html
import com.interactivemesh.j3d.community.utils.navigation.orbit.OrbitBehaviorInterim;

// JavaFX
import javafx.async.RunnableFuture;

/**
 * CharacterCubeUniverse.java
 *
 * Version: 5.1
 * Date: 2010/09/19
 *
 * Copyright (c) 2009-2010
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * but you may not modify, compile, or distribute this 'CharacterCubeUniverse.java'.
 *
 */
final class CharacterCubeUniverse implements RunnableFuture {

    static {
        System.out.println("FXCharacterCube : Copyright (c) 2009-2010 August Lammersdorf, www.InteractiveMesh.com.");
    }

    private BoundingSphere      globalBounds        = 	null;

    private View                view                = 	null;
    private Locale              locale              = 	null;

    private Background          background          =   null;

    private OrbitBehaviorInterim orbitBehInterim    = 	null;

    private PickCanvas          pickCanvas          = 	null;

    private BranchGroup         sceneBranch         =   null;
    private BranchGroup         viewBranch          = 	null;
    private BranchGroup         enviBranch          = 	null;
    
    private Alpha               rotationAlpha       =   null;
    private long                rotationStartTime   =   0;

    private Switch              cubeSwitch          =   null;
    private TransparencyAttributes cubeTransAttr    =   null;

    private Color               bgColor0            =   new Color(0, 102, 204); // RGB 0.0f, 0.4f, 0.8f, HSB 209, 100,  80
    private Color               bgColor1            =   new Color(0, 153, 255); // RGB 0.0f, 0.6f, 1.0f, HSB 203, 100, 100

    private UpdateBehavior      updateBehavior      =   null;
    private VantagePointBehavior vpExecutor         =	null;
    private LinkedHashMap<String, VantagePoint> vantagepointHM = new LinkedHashMap<String, VantagePoint>();
    
    private ScriptUpdater     scriptUpdater       =   null;

    //private int sumTriangles = 0;

    CharacterCubeUniverse() {
    }
    
    // JavaFX Interface RunnableFuture
    public void run() {
        initUniverse();
    }

    // Creates and returns the lightweight 3D canvas
    FXCanvas3DSB createFXCanvas3D(FXCanvas3DRepainter repainter, JPanel parent) {

        FXCanvas3DSB fxCanvas3D = null;

        try {
            GraphicsConfigTemplate3D gCT = new GraphicsConfigTemplate3D();

            fxCanvas3D = new FXCanvas3DSB(gCT);
            fxCanvas3D.setRepainter(repainter);
        }
        catch (Exception e) {
            System.out.println("CharacterCubeUniverse: FXCanvas3D failed !!");
            e.printStackTrace();
            System.exit(0);
        }

        createBackgroundImage(fxCanvas3D);
        // Optional: less 'flickering' during resizing
        fxCanvas3D.setBackground(bgColor0);

        // Due to Java 3D's implementation of off-screen rendering:
        // 1. Set size
        // 2. Provide a parent
        // 3. Get the heavyweight off-screen Canvas3D and add this to the view object

        // 1. Size
        Dimension dim = parent.getSize();
        if (dim.width < 1 || dim.height < 1) {
            dim.width = 100;
            dim.height = 100;
            parent.setSize(dim);
        }
        parent.setPreferredSize(dim);
        fxCanvas3D.setPreferredSize(dim);
        fxCanvas3D.setSize(dim);

        // 2. Parent
        // BorderLayout
        parent.setLayout(new BorderLayout());
        parent.add(fxCanvas3D, BorderLayout.CENTER);

        // 3. Heavyweight off-screen Canvas3D of the lightweight FXCanvas3D
        Canvas3D offCanvas3D = fxCanvas3D.getOffscreenCanvas3D();
        if (offCanvas3D != null) {
            // View renders into the off-screen Canvas3D
            view.addCanvas3D(offCanvas3D);

            // PickCanvas operates on the not visible off-screen Canvas3D
    	    pickCanvas = new PickCanvas(offCanvas3D, sceneBranch);
    	    pickCanvas.setMode(PickInfo.PICK_GEOMETRY);
    	    pickCanvas.setFlags(PickInfo.NODE);
    	    pickCanvas.setTolerance(4.0f);
        }
        else {
            System.out.println("CharacterCubeUniverse: Off-screen Canvas3D = null !!");
            System.exit(0);
        }

        // MouseAdapter for picking
        MouseListener mouseAdapter = new MouseListener() {
            public void mouseClicked(MouseEvent event) {
                int clicks = event.getClickCount();
                if (clicks == 2 && SwingUtilities.isLeftMouseButton(event)) {
                    changeRotationCenter(event);
                }
            }
            public void mouseEntered(MouseEvent e) {}
            public void mouseExited(MouseEvent e) {}
            public void mousePressed(MouseEvent e) {}
            public void mouseReleased(MouseEvent e) {}
        };

        fxCanvas3D.addMouseListener(mouseAdapter);

        // Navigator
        setupNavigator(fxCanvas3D);

        return fxCanvas3D;
    }

    //
    // Scrict update
    //

    void setScriptUpdater(ScriptUpdater updater) {
        scriptUpdater = updater;
    }

    //
    // Scene interaction
    //
    
    void setMinFrameCycleTime(long minimumTime) {
        if (view != null)
            view.setMinimumFrameCycleTime(minimumTime);
    }

    void setProjectionMode(String mode) {
        int projMode = View.PERSPECTIVE_PROJECTION;
        if (mode.equalsIgnoreCase("Parallel"))
            projMode = View.PARALLEL_PROJECTION;
        orbitBehInterim.setProjectionMode(projMode);
    }

    void setVantagePoint(String name) {
        VantagePoint vp = vantagepointHM.get(name);
        vpExecutor.setVP(vp);
        vpExecutor.postId(VantagePointBehavior.APPLY_VP);
    }

    void setCubeTransparency(int transparency) {
        if (cubeTransAttr == null)
            return;

        if (transparency > 100)
            transparency = 100;
        else if (transparency < 0)
            transparency = 0;
        
        float value = (float)transparency/100.0f;

        if (value > 0.98) {
            value = 1.0f;
            cubeSwitch.setWhichChild(Switch.CHILD_NONE);
        }
        else if (cubeSwitch.getWhichChild() != Switch.CHILD_ALL) {
            cubeSwitch.setWhichChild(Switch.CHILD_ALL);
        }
        
        cubeTransAttr.setTransparency(value);
    }

    void resetRotation() {
        rotationAlpha.pause(rotationStartTime);
    }

    // Rotation speed and direction
    // Update via a Behavior provides smoother animation
    void setRotationSpeed(int speed) {
        updateBehavior.setSpeed(speed);
        updateBehavior.postId(UpdateBehavior.ROT_SPEED);
    }
    private void updateRotationSpeed(int speed) {
        if (rotationAlpha == null)
            return;

        if (speed > 99)
            speed = 99;
        else if (speed < 1)
            speed = 1;

        // Loop duration determines rotation speed
        long duration = 0;
        long pauseTime = 0;
        float pauseValue = 0;

        if (speed > 52) {

            // isPaused
            if (rotationAlpha.isPaused()) {

                if (rotationAlpha.getMode() == Alpha.INCREASING_ENABLE) {
                    rotationAlpha.resume();
                    return;
                }

                pauseTime = rotationAlpha.getPauseTime();
            }
            // Pause
            else {
                pauseTime = System.currentTimeMillis();
                rotationAlpha.pause(pauseTime);
            }

            pauseValue = rotationAlpha.value();

            if (rotationAlpha.getMode() != Alpha.INCREASING_ENABLE)
                rotationAlpha.setMode(Alpha.INCREASING_ENABLE);

            // New IncreasingAlphaDuration
//          duration = (long)1000 *((100 - speed));                 // [47, 1] non linear
            duration = (long)(2000 * ( ( 47.0/(speed - 52) ) ));    // [47, 1] linear

            // Offset according to rotationAlpha's pauseValue and the new IncreasingAlphaDuration
            long resumeOffsetTime = (long)(pauseValue * duration);

            rotationAlpha.setIncreasingAlphaDuration(duration);

            // Resume

            // Alpha source code: newStartTime = oldStartTime + resumeTime - pauseTime
            // Start immediately and adapt new duration:
            //   => System.currentTimeMillis() - resumeOffsetTime = oldStartTime + resumeTime - pauseTime
            //   => resumeTime = System.currentTimeMillis() - resumeOffsetTime - oldStartTime + pauseTime

            long resumeTime = System.currentTimeMillis() - resumeOffsetTime
                              - rotationAlpha.getStartTime() + pauseTime;

            rotationAlpha.resume(resumeTime);
        }
        else if (speed < 48) {

            // isPaused
            if (rotationAlpha.isPaused()) {

                if (rotationAlpha.getMode() == Alpha.DECREASING_ENABLE) {
                    rotationAlpha.resume();
                    return;
                }

                pauseTime = rotationAlpha.getPauseTime();
            }
            // Pause
            else {
                pauseTime = System.currentTimeMillis();
                rotationAlpha.pause(pauseTime);
            }

            pauseValue = rotationAlpha.value();

            if (rotationAlpha.getMode() != Alpha.DECREASING_ENABLE)
                rotationAlpha.setMode(Alpha.DECREASING_ENABLE);

            // New DecreasingAlphaDuration
//          duration = (long)(1000 * speed);                        // [47, 1] non linear
            duration = (long)(2000 * ( ( 47.0/(48 - speed) ) ));    // [47, 1] linear

            // Offset according to pauseValue and the new DecreasingAlphaDuration
            long resumeOffsetTime = (long)((1-pauseValue) * duration);

            rotationAlpha.setDecreasingAlphaDuration(duration);

            // Resume
            rotationAlpha.resume(System.currentTimeMillis() - resumeOffsetTime
                                 - rotationAlpha.getStartTime() + pauseTime);
        }
        else {
            if (!rotationAlpha.isPaused()) {
                rotationAlpha.pause(System.currentTimeMillis());
            }
        }
    }

    void closeUniverse() {
        view.removeAllCanvas3Ds();
        view.attachViewPlatform(null);
        locale.getVirtualUniverse().removeAllLocales();
    }

    //
    // Create universe
    //
    private void initUniverse() {

        createVantagePoints();

        createUniverse();
        createScene();

        setLive();
    }

    // Setup navigation
    private void setupNavigator(FXCanvas3DSB component) {

        orbitBehInterim.setAWTComponent(component);

        double sceneRadius = 1.0f;

        Bounds bounds = sceneBranch.getBounds();
        BoundingSphere sphereBounds = null;

        if (bounds.isEmpty()) {
            sphereBounds = new BoundingSphere();
        }
        else {
            if (bounds instanceof BoundingSphere)
                sphereBounds = (BoundingSphere)bounds;
            else
                sphereBounds = new BoundingSphere(bounds);
        }

        sceneRadius = sphereBounds.getRadius();

        orbitBehInterim.setTransFactors(sceneRadius/4.0f, sceneRadius/4.0f);
        orbitBehInterim.setZoomFactor(sceneRadius/4.0f);
        orbitBehInterim.setRotFactors(0.5f, 0.5f);

        orbitBehInterim.setClippingBounds(sphereBounds);

        setVantagePoint("Home");
    }

    private void createScene() {

        // Font
        Font font = new Font("Lucida Sans", Font.PLAIN, 100);

        // Extrusion
        GeneralPath extrPath = new GeneralPath();

        float extend = 1.0f;
        float depth = 20.0f;
        float cut = depth/10.0f;

        extrPath.moveTo(0.0f, 0.0f);
        extrPath.lineTo(cut, extend);
        extrPath.lineTo(depth-cut, extend);
        extrPath.lineTo(depth, 0.0f);

        // Factories
        AWTShapeExtrusion   extrusion   =   new AWTShapeExtrusion(extrPath);
        AWTShapeExtruder    extruder    =   new AWTShapeExtruder(0.12, extrusion, Math.toRadians(24));
        String3D            string3D    =   new String3D(font, extruder);

        string3D.setAlignment(String3D.Alignment.CENTER);
        string3D.setPath(String3D.Path.RIGHT);

        // Characters' Appearances for each cube face
        Appearance[] charAppears = createAppearances(5);

        //
        Group charGroup = new Group();

        TransformGroup targetTG = null;
        Transform3D targetTransform = null;

        // Arrows 
        
        int[] arrows = {8608, 8646, 8673, 8611, 8651,
                        8613, 8614, 8670, 8632, 8676,
                        8618, 8605, 8620, 8621, 8668,
                        8679, 8653, 8661, 8666, 8665,
                        8634, 8629, 8630, 8625, 8623
        };
        
        targetTG = new TransformGroup();
        targetTransform = new Transform3D();
        targetTransform.setRotation(new AxisAngle4d(0, 1, 0, Math.toRadians(180)));
        targetTransform.setTranslation(new Vector3d(700-90, 50, -50)); // Back
        targetTG.setTransform(targetTransform);
        addCharacters(arrows, charAppears, string3D, targetTG);
        charGroup.addChild(targetTG);

        // Dingbats

        int[] bats = { 9986,  9990,  9991,  9992,  9988,  
                       9998,  9997, 10001, 10002,  9993,
                      10006, 10007, 10070, 10012, 10004,
                      10051, 10053, 10021, 10052, 10057,
                      10102, 10113, 10104, 10115, 10106
        };

        targetTG = new TransformGroup();
        targetTransform = new Transform3D();
        targetTransform.setTranslation(new Vector3d(90, 50, 700+50)); // Front
        targetTG.setTransform(targetTransform);
        addCharacters(bats, charAppears, string3D, targetTG);
        charGroup.addChild(targetTG);
        
        // Latin
        
        int[] latins = {  65,  66,  77,  89,  90,
                          33,  37,  38,  35,  63,
                         123,  43,  64,  47,  93,
                         167, 181, 164, 169, 182,
                         283, 339, 294, 373, 367
        };
        
        targetTG = new TransformGroup();
        targetTransform = new Transform3D();
        targetTransform.setRotation(new AxisAngle4d(0, 1, 0, Math.toRadians(90)));
        targetTransform.setTranslation(new Vector3d(700+50, 50, 610)); // Right
        targetTG.setTransform(targetTransform);
        addCharacters(latins, charAppears, string3D, targetTG);
        charGroup.addChild(targetTG);

        // Mathematical Operators

        int[] mathOps = { 8704, 8707, 8709, 8721, 8748,
                          8771, 8777, 8781, 8763, 8783,
                          8811, 8807, 8812, 8804, 8810,
                          8834, 8844, 8833, 8847, 8837,
                          8853, 8856, 8857, 8860, 8862
        };

        targetTG = new TransformGroup();
        targetTransform = new Transform3D();
        targetTransform.setTranslation(new Vector3d(-50, 50, 90));  // Left
        targetTransform.setRotation(new AxisAngle4d(0, 1, 0, Math.toRadians(-90)));
        targetTG.setTransform(targetTransform);
        addCharacters(mathOps, charAppears, string3D, targetTG);
        charGroup.addChild(targetTG);

        //
        // Cube
        //
        GeometryArray cubeGeom = CubeGeometry.create(GeometryArray.COORDINATES |
                                                     GeometryArray.NORMALS |
                                                     GeometryArray.COLOR_3, 
                                                     700);
        Appearance cubeAppear = new Appearance();

        Material material = new Material();
        material.setColorTarget(Material.DIFFUSE);  // default
        material.setAmbientColor(0, 0, 0);
        material.setEmissiveColor(0, 0, 0);         // default
        material.setSpecularColor(0, 0, 0);
        cubeAppear.setMaterial(material);

        PolygonAttributes polyAttri = new PolygonAttributes();
        polyAttri.setBackFaceNormalFlip(true);
        polyAttri.setCullFace(PolygonAttributes.CULL_NONE);
        cubeAppear.setPolygonAttributes(polyAttri);

        cubeTransAttr = new TransparencyAttributes();
        cubeTransAttr.setCapability(TransparencyAttributes.ALLOW_VALUE_WRITE);
        cubeTransAttr.setTransparencyMode(TransparencyAttributes.NICEST);
        cubeTransAttr.setTransparency(0.3f); // Initial value
        cubeAppear.setTransparencyAttributes(cubeTransAttr);

        Shape3D cubeShape3D = new Shape3D(cubeGeom, cubeAppear);

        // Hide fully transparent cube
        cubeSwitch = new Switch();
        cubeSwitch.setCapability(Switch.ALLOW_SWITCH_READ);
        cubeSwitch.setCapability(Switch.ALLOW_SWITCH_WRITE);
        cubeSwitch.setWhichChild(Switch.CHILD_ALL);

        cubeSwitch.addChild(cubeShape3D);

        // Rotation

        TransformGroup rotationGroup = new TransformGroup();
        rotationGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);

        rotationAlpha = new Alpha();
        rotationAlpha.setLoopCount(-1);
        rotationAlpha.setIncreasingAlphaDuration(0);

        // Initialize rotation        
        rotationAlpha.setIncreasingAlphaDuration(2000 * 47);    // start speed = min speed
        rotationStartTime = System.currentTimeMillis();         // start time
        rotationAlpha.setStartTime(rotationStartTime);          // start
        rotationAlpha.pause(rotationStartTime);                 // pause at start time

        RotationInterpolator rotInterp = new RotationInterpolator(rotationAlpha, rotationGroup);
        rotInterp.setSchedulingBounds(globalBounds);

        // Center cube at (0,0,0)
        TransformGroup positionGroup = new TransformGroup();
        Transform3D rotT3D = new Transform3D();
        rotT3D.setTranslation(new Vector3d(-350, -350, -350));
        positionGroup.setTransform(rotT3D);

        // Order opaque/transparency shapes
        OrderedGroup orderedGroup = new OrderedGroup();
        orderedGroup.addChild(charGroup);
        orderedGroup.addChild(cubeSwitch);

        positionGroup.addChild(orderedGroup);
        rotationGroup.addChild(positionGroup);
        
        sceneBranch.addChild(rotationGroup);
        sceneBranch.addChild(rotInterp);

//System.err.println("sumTriangles = " + (sumTriangles/3));
    }

    private void addCharacters(int[] characters, Appearance[] charAppears,
                               String3D string3D, TransformGroup targetTG) {

        TransformGroup tg = null;
        Transform3D t3d = new Transform3D();
        Vector3d translation = new Vector3d(0, 0, 0);

        GeometryArray geom = null;

        int k = 0;

        for (int i=0; i < 5; i++) {
            for (int j=0; j < 5; j++) {

                geom = string3D.getStringGeometry(new String(characters, k, 1));

//sumTriangles += geom.getVertexCount();

                tg = new TransformGroup();
                t3d.setTranslation(translation);
                tg.setTransform(t3d);

                tg.addChild(new Unicode3D(geom, charAppears[k], characters[k]));
                targetTG.addChild(tg);

                k++;
                translation.x += 130;
            }

            translation.x = 0;
            translation.y += 130;
        }
    }

    final private class Unicode3D extends Shape3D {
        private int unicodeCodePoint = -1;
        private Unicode3D(GeometryArray geom, Appearance appear, int codepoint) {
            super(geom, appear);
            unicodeCodePoint = codepoint;
        }
    }

    private Appearance[] createAppearances(int dim) {

        // Appearances
        Appearance[] appears = new Appearance[dim*dim];

        int k = 0;
        int direction = 1;
        float h = 0;

        float step = 1.0f / (dim*dim);

        for (int i=0; i < dim; i++) {
            for (int j=0; j < dim; j++) {

                appears[k] = new Appearance();
                appears[k].setMaterial(getMaterial(h));
                h += step * direction;

                k++;
            }

            direction *= (-1);
            if (direction < 0)
                h += step * (dim-1);
            else
                h += step * (dim+1);
        }

        return appears;
    }

    private Material getMaterial(float hue) {
        Material mat = new Material();
        mat.setAmbientColor(0.0f, 0.0f, 0.0f);
        mat.setDiffuseColor(new Color3f(new Color(Color.HSBtoRGB(hue, 0.85f, 0.7f))));
        mat.setEmissiveColor(new Color3f(new Color(Color.HSBtoRGB(hue, 0.85f, 0.3f))));
        mat.setSpecularColor(0.0f, 0.0f, 0.0f);
        mat.setShininess(64f);
        return mat;
    }

    private void changeRotationCenter(MouseEvent event) {

        pickCanvas.setShapeLocation(event.getX(), event.getY());
        PickInfo pickInfo = pickCanvas.pickClosest();

        if (pickInfo != null) {

            Shape3D pickedShape3D = (Shape3D)pickInfo.getNode();

            Transform3D locToVWord = new Transform3D();
            pickedShape3D.getLocalToVworld(locToVWord);

            Point3d rotationPoint = new Point3d();

            Bounds bounds = pickedShape3D.getBounds();

            if (bounds instanceof BoundingBox) {
                BoundingBox pickedBox = (BoundingBox)bounds;

                Point3d lower = new Point3d();
                Point3d upper = new Point3d();
                pickedBox.getLower(lower);
                pickedBox.getUpper(upper);

                rotationPoint.set(lower.x + (upper.x - lower.x)/2,
                                  lower.y + (upper.y - lower.y)/2,
                                  lower.z + (upper.z - lower.z)/2);
            }
            else {
                BoundingSphere pickedSphere = null;
                if (bounds instanceof BoundingSphere)
                    pickedSphere = (BoundingSphere)bounds;
                else
                    pickedSphere = new BoundingSphere(bounds);

                pickedSphere.getCenter(rotationPoint);
            }

            if (pickedShape3D instanceof Unicode3D) {
                scriptUpdater.setUnicodeCodePoint(((Unicode3D)pickedShape3D).unicodeCodePoint);
            }

            locToVWord.transform(rotationPoint);
            orbitBehInterim.setRotationCenter(rotationPoint, true); // isLookAtRotCenter
        }
    }
    // Set live
    private void setLive() {
	sceneBranch.compile();
        locale.addBranchGraph(sceneBranch);
        locale.addBranchGraph(viewBranch);
        locale.addBranchGraph(enviBranch);
    }

	// Base world
    private void createUniverse() {

        // Bounds
    	globalBounds = new BoundingSphere();
        globalBounds.setRadius(Double.MAX_VALUE);

        //
        // Viewing
        //
        view = new View();
        view.setPhysicalBody(new PhysicalBody());
        view.setPhysicalEnvironment(new PhysicalEnvironment());
        // works fine for all values
        //view.setMinimumFrameCycleTime(30);

        //
        // SuperStructure
        //
        VirtualUniverse vu = new VirtualUniverse();
        locale = new Locale(vu);
        //
        // BranchGraphs
        //
        sceneBranch = new BranchGroup();
        viewBranch = new BranchGroup();
        enviBranch = new BranchGroup();

        // ViewBranch

        TransformGroup viewTG = new TransformGroup();
        viewTG.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);

        ViewPlatform vp = new ViewPlatform();
        view.attachViewPlatform(vp);

        orbitBehInterim = new OrbitBehaviorInterim(OrbitBehaviorInterim.REVERSE_ALL);
        orbitBehInterim.setViewingTransformGroup(viewTG);
        orbitBehInterim.setVpView(view);
        orbitBehInterim.setSchedulingBounds(globalBounds);
        orbitBehInterim.setClippingEnabled(true);
        orbitBehInterim.setProjectionMode(View.PERSPECTIVE_PROJECTION);
        orbitBehInterim.setPureParallelEnabled(true);

        updateBehavior = new UpdateBehavior();
        updateBehavior.setSchedulingBounds(globalBounds);

        vpExecutor = new VantagePointBehavior(orbitBehInterim);
        vpExecutor.setSchedulingBounds(globalBounds);

        DirectionalLight headLight = new DirectionalLight();
        headLight.setInfluencingBounds(globalBounds);

        viewTG.addChild(vp);
        viewTG.addChild(orbitBehInterim);
        viewTG.addChild(updateBehavior);
        viewTG.addChild(vpExecutor);
        viewTG.addChild(headLight);

        viewBranch.addChild(viewTG);

        // EnviBranch

        background = new Background();
        background.setCapability(Background.ALLOW_IMAGE_WRITE);
        background.setApplicationBounds(globalBounds);
        background.setColor(new Color3f(bgColor0));

        enviBranch.addChild(background);
    }

    // Image for Background node
    private void createBackgroundImage(final FXCanvas3DSB canvas3D) {
        canvas3D.addComponentListener(new ComponentAdapter() {
            @Override
            public void componentResized(ComponentEvent e) {
                int width = canvas3D.getWidth();
                int height = canvas3D.getHeight();

                ImageComponent2D currImage = background.getImage();
                if (currImage != null) {
                    if (currImage.getWidth() == width && currImage.getHeight() == height)
                        return;
                }

                BufferedImage gradientImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
                Graphics2D g2d = (Graphics2D)gradientImage.getGraphics();

                int splitHeight = (int)(height/5f); // yUp !

                GradientPaint gp = new GradientPaint(0, 0,           bgColor0,
                                                     0, splitHeight, bgColor1);
                g2d.setPaint(gp);
                g2d.fillRect(0, 0, width, splitHeight);

                gp = new GradientPaint(0, splitHeight, bgColor1,
                                       0, height,      bgColor0);
                g2d.setPaint(gp);
                g2d.fillRect(0, splitHeight, width, height);

                g2d.dispose();

                background.setImage(new ImageComponent2D(ImageComponent2D.FORMAT_RGB, gradientImage, true, true));

                /*
                int splitHeight = (int)(height*4/5f);

                GradientPaint gp = new GradientPaint(0, 0,           bgColor0,
                                                     0, splitHeight, bgColor1);
                g2d.setPaint(gp);
                g2d.fillRect(0, 0, width, splitHeight);

                gp = new GradientPaint(0, splitHeight, bgColor1,
                                       0, height,      bgColor0);
                g2d.setPaint(gp);
                g2d.fillRect(0, splitHeight, width, height);

                g2d.dispose();

                bg.setImage(new ImageComponent2D(ImageComponent2D.FORMAT_RGB, gradientImage));
                */
            }
        });
    }

    private void createVantagePoints() {
        Point3d  lookAt    = new Point3d(0, 0, 0);
        Vector3d upVector  = new Vector3d(0.0, 1.0, 0.0);
        Point3d  rotCenter = new Point3d(0, 0, 0);
        double   fov       = Math.toRadians(45); // default value is used
        //                name    -    VP's position parall.    -    VP's position persp.   -   look at - up vector - center of rotation - fov angle
        new VantagePoint("Front",  new Point3d(0, 0, 1500),       new Point3d(0, 0, 1780),       lookAt, upVector, rotCenter, fov);
        new VantagePoint("Back",   new Point3d(0, 0, -1500),      new Point3d(0, 0, -1780),      lookAt, upVector, rotCenter, fov);
        new VantagePoint("Left",   new Point3d(-1500, 0, 0),      new Point3d(-1780, 0, 0),      lookAt, upVector, rotCenter, fov);
        new VantagePoint("Right",  new Point3d(1500, 0, 0),       new Point3d(1780, 0, 0),       lookAt, upVector, rotCenter, fov);
        new VantagePoint("Home",   new Point3d(1300, 900, 1300), new Point3d(1300, 900, 1300), lookAt, upVector, rotCenter, fov);
        new VantagePoint("Center", new Point3d(0, 0, 0),          new Point3d(0, 0, 0), new Point3d(0, 0, -1), upVector, rotCenter,  fov);
    }

    private final class VantagePoint {

        private String      name        =   null;

	private Point3d     eyeParallel =   new Point3d();
        private Point3d     eyePerspect =   new Point3d();
	private Point3d     viewCenter 	=   new Point3d();
	private Vector3d    up          =   new Vector3d();
	private Point3d     rotCenter 	=   new Point3d();
	//private double      fov	=   Math.PI/4;

	private VantagePoint(String name, Point3d eyePar, Point3d eyePer, Point3d viewCenter, Vector3d up, Point3d rotationCenter, double fov) {

            this.name = name;

            this.eyeParallel.set(eyePar);
            this.eyePerspect.set(eyePer);
            this.viewCenter.set(viewCenter);
            this.up.set(up);
            this.rotCenter.set(rotationCenter);
            //this.fov = fov;

            vantagepointHM.put(name, this);
        }

        @Override
        public String toString() {
            return name;
        }

        void applyTo(OrbitBehaviorInterim navigator) {
            
            if (orbitBehInterim.getProjectionMode() == View.PARALLEL_PROJECTION) {
                navigator.setViewingTransform(eyeParallel, viewCenter, up, rotCenter);
            }
            else {
                navigator.setViewingTransform(eyePerspect, viewCenter, up, rotCenter);
            }
			
            //navigator.setFieldOfView(fov);
        }
    }

    // Sets vantage point in behavior scheduler
    private final class VantagePointBehavior extends Behavior {

        private static final int        APPLY_VP = 1;
        private WakeupOnBehaviorPost    post = new WakeupOnBehaviorPost(this, APPLY_VP);

        private OrbitBehaviorInterim 	orbitBeh = null;
        private VantagePoint            vantagePoint = null;

        private VantagePointBehavior(OrbitBehaviorInterim orbitBeh) {
            this.orbitBeh = orbitBeh;
        }

        VantagePoint getVP() {
            return vantagePoint;
        }

        void setVP(VantagePoint vp) {
            vantagePoint = vp;
        }

        @Override
        public void initialize() {
            wakeupOn(post);
        }
        @Override
        public void processStimulus(Enumeration criteria) {
            if (vantagePoint != null) {
                if (rotationAlpha.isPaused())
                    rotationAlpha.pause(rotationStartTime);
                vantagePoint.applyTo(orbitBeh);
            }
            vantagePoint = null;
            wakeupOn(post);
        }
    }

    // Rotation speed update
    private final class UpdateBehavior extends Behavior {

        private static final int        ROT_SPEED   =   1;
        private WakeupOnBehaviorPost    updatePost  =   new WakeupOnBehaviorPost(this, ROT_SPEED);
        private int                     speed       =   50;

        private WakeupCriterion         criterion   =   null;
        private int                     postID      =   Integer.MIN_VALUE;

        private void setSpeed(int speed) {
            this.speed = speed;
        }

        @Override
        public void initialize() {
            wakeupOn(updatePost);
        }
        @Override
        public void processStimulus(Enumeration criteria) {

            while (criteria.hasMoreElements()) {
                criterion = (WakeupCriterion)criteria.nextElement();
                if (criterion instanceof WakeupOnBehaviorPost) {
                    postID = ((WakeupOnBehaviorPost)criterion).getTriggeringPostId();
                    if (postID == ROT_SPEED) {
                        updateRotationSpeed(speed);
                    }
                }
            }

            wakeupOn(updatePost);
        }
    }

}