/*
 * org.jdesktop.j3d.examples.distort_glyph.DistortBehavior.java / gold.jpg
 *
 * Copyright (c) 2007 Sun Microsystems, Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistribution of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * - Redistribution in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the
 *   distribution.
 *
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * This software is provided "AS IS," without a warranty of any
 * kind. ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND
 * WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY
 * EXCLUDED. SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL
 * NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF
 * USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
 * DERIVATIVES. IN NO EVENT WILL SUN OR ITS LICENSORS BE LIABLE FOR
 * ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE THIS SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * You acknowledge that this software is not designed, licensed or
 * intended for use in the design, construction, operation or
 * maintenance of any nuclear facility.
 *
 * $Revision: 1.2 $
 * $Date: 2007/02/09 17:21:36 $
 * $State: Exp $
 */
package com.interactivemesh.j3d.testspace.jfx.distortstring;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;

import java.awt.geom.GeneralPath;

import java.io.IOException;

import java.util.Enumeration;
import java.util.Map;

import javax.media.j3d.Alpha;
import javax.media.j3d.Appearance;
import javax.media.j3d.Background;
import javax.media.j3d.Behavior;
import javax.media.j3d.BoundingBox;
import javax.media.j3d.BoundingSphere;
import javax.media.j3d.Bounds;
import javax.media.j3d.BranchGroup;
import javax.media.j3d.Canvas3D;
import javax.media.j3d.DirectionalLight;
import javax.media.j3d.Geometry;
import javax.media.j3d.GeometryArray;
import javax.media.j3d.GeometryUpdater;
import javax.media.j3d.GLSLShaderProgram;
import javax.media.j3d.GraphicsConfigTemplate3D;
import javax.media.j3d.Locale;
import javax.media.j3d.Material;
import javax.media.j3d.PhysicalBody;
import javax.media.j3d.PhysicalEnvironment;
import javax.media.j3d.Shader;
import javax.media.j3d.ShaderAppearance;
import javax.media.j3d.ShaderAttributeSet;
import javax.media.j3d.ShaderAttributeValue;
import javax.media.j3d.ShaderProgram;
import javax.media.j3d.Shape3D;
import javax.media.j3d.SourceCodeShader;
import javax.media.j3d.Switch;
import javax.media.j3d.TexCoordGeneration;
import javax.media.j3d.Texture;
import javax.media.j3d.TextureAttributes;
import javax.media.j3d.Transform3D;
import javax.media.j3d.TransformGroup;
import javax.media.j3d.TriangleArray;
import javax.media.j3d.View;
import javax.media.j3d.ViewPlatform;
import javax.media.j3d.VirtualUniverse;
import javax.media.j3d.WakeupCriterion;
import javax.media.j3d.WakeupOnBehaviorPost;
import javax.media.j3d.WakeupOnElapsedFrames;
import javax.media.j3d.WakeupOr;

import javax.swing.JPanel;

import javax.vecmath.AxisAngle4d;
import javax.vecmath.Color3f;
import javax.vecmath.Point3d;
import javax.vecmath.Point3f;
import javax.vecmath.Vector3d;
import javax.vecmath.Vector3f;
import javax.vecmath.Vector4f;

import com.sun.j3d.utils.image.TextureLoader;
import com.sun.j3d.utils.shader.StringIO;

// FXCanvas3D API 3.0, see http://www.interactivemesh.org/testspace/j3dmeetsjfx.html
import com.interactivemesh.j3d.community.gui.FXCanvas3DSB;
import com.interactivemesh.j3d.community.gui.FXCanvas3DRepainter;

// AWTShapeExtruder 3.0, see http://www.interactivemesh.org/testspace/awtshapeextruder.html
import com.interactivemesh.j3d.community.utils.geometry.AWTShapeExtruder;
import com.interactivemesh.j3d.community.utils.geometry.AWTShapeExtrusion;
import com.interactivemesh.j3d.community.utils.geometry.String3D;

// OrbitBehaviorInterim 2.1, see http://www.interactivemesh.org/testspace/orbitbehavior.html
import com.interactivemesh.j3d.community.utils.navigation.orbit.OrbitBehaviorInterim;

// JavaFX
import javafx.async.RunnableFuture;

/**
 * DistortStringUniverse.java
 *
 * Version: 6.1
 * Date: 2010/09/19
 *
 * Copyright (c) 2009-2010
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 *
 * under consideration of the copyrigth notice above for the 
 * internal class 'DistortBehavior' and the image 'gold.jpg',
 *
 * but you may not modify, compile, or distribute this 'DistortStringUniverse.java'.
 *
 */
final class DistortStringUniverse implements RunnableFuture {

    static {
        System.out.println("FXDistortString : Copyright (c) 2009-2010 August Lammersdorf, www.InteractiveMesh.com.");
    }

    private BoundingSphere      globalBounds        = 	null;

    private View                view                = 	null;
    private Locale              locale              = 	null;

    private Canvas3D            offCanvas3D         =   null;

    private BranchGroup         sceneBranch         = 	null;
    private BranchGroup         viewBranch          = 	null;
    private BranchGroup         enviBranch          = 	null;

    private OrbitBehaviorInterim orbitBehInterim    = 	null;

    private DistortBehavior     distortBeh          =   null;

    private Switch              shapeSwitch         =   null;
    private static final int    GOLD_SHAPE3D        =   0;
    private static final int    WOOD_SHAPE3D        =   1;

    private Appearance          goldAppearance      =   null;
    private TextureAttributes   texAttr             =   null;
    private TexCoordGeneration  texCoordGenLinear   =   null;
    private TexCoordGeneration  texCoordGenSphere   =   null;

    private ShaderAttributeValue attrValueSpx       =   null;
    private ShaderAttributeValue attrValueSpy       =   null;
    private ShaderAttributeValue attrValueDistTex   =   null;
    
    private boolean             isGLSL              =   false;
    private boolean             isShader            =   false;

    private Color               bgColor             = 	new Color(0.0f, 0.4f, 0.8f);

    DistortStringUniverse() {
    }

    // JavaFX Interface RunnableFuture
    @Override
    public void run() {
        initUniverse();
    }

    // Creates and returns the lightweight 3D canvas
    FXCanvas3DSB createFXCanvas3D(FXCanvas3DRepainter repainter, JPanel parent) {

        FXCanvas3DSB fxCanvas3D = null;

        try {
            GraphicsConfigTemplate3D gCT = new GraphicsConfigTemplate3D();

            fxCanvas3D = new FXCanvas3DSB(gCT);
            fxCanvas3D.setRepainter(repainter);
        }
        catch (Exception e) {
            System.out.println("DistortStringUniverse: FXCanvas3D failed !!");
            e.printStackTrace();
            System.exit(0);
        }

        // Optional: less 'flickering' during resizing
        fxCanvas3D.setBackground(bgColor);

        // Due to Java 3D's implementation of off-screen rendering:
        // 1. Set size
        // 2. Provide a parent
        // 3. Get the heavyweight off-screen Canvas3D and add this to the view object

        // 1. Size
        Dimension dim = parent.getSize();
        if (dim.width < 1 || dim.height < 1) {
            dim.width = 100;
            dim.height = 100;
            parent.setSize(dim);
        }
        parent.setPreferredSize(dim);
        fxCanvas3D.setPreferredSize(dim);
        fxCanvas3D.setSize(dim);

        // 2. Parent
        // BorderLayout
        parent.setLayout(new BorderLayout());
        parent.add(fxCanvas3D, BorderLayout.CENTER);

        // 3. Heavyweight off-screen Canvas3D of the lightweight FXCanvas3D
        offCanvas3D = fxCanvas3D.getOffscreenCanvas3D();
        if (offCanvas3D != null) {
            // View renders into the off-screen Canvas3D
            view.addCanvas3D(offCanvas3D);
        }
        else {
            System.out.println("DistortStringUniverse: Off-screen Canvas3D is null !!");
            System.exit(0);
        }

        // Navigator
        setupNavigator(fxCanvas3D);

        return fxCanvas3D;
    }

    //
    // Scene interaction
    //
    void setVantagePoint(String vp) {
        Transform3D vpTransform = new Transform3D();
        if (vp.equalsIgnoreCase("Front")) {
            vpTransform.setTranslation(new Vector3d(0.0, 0.45, 4.0));
        }
        else if (vp.equalsIgnoreCase("Side")) {
            vpTransform.setRotation(new AxisAngle4d(0, 1, 0, Math.toRadians(-90)));
            vpTransform.setTranslation(new Vector3d(-4.0, 0.45, 0.0));
        }
        else
            return;
        orbitBehInterim.setViewingTransform(vpTransform, new Point3d());
    }

    void startStopDistort(boolean start) {
        distortBeh.startStop(start);
    }

    void setDistortSpeed(int s) {
        distortBeh.speed(s);
    }

    // Shader supported ?
    boolean isGLSLavailable() {
        if (offCanvas3D == null)
            throw new IllegalStateException("isGLSLavailable : No Canvas3D added !!");

        Map c3dProps = offCanvas3D.queryProperties();

        isGLSL = ((Boolean)c3dProps.get("shadingLanguageGLSL"));

        return isGLSL;
    }
    
    void setGoldTexture(String mode) {
        if (mode.equalsIgnoreCase("A")) {
            goldAppearance.setTexCoordGeneration(texCoordGenSphere);
            texAttr.setTextureMode(TextureAttributes.REPLACE);
        }
        else if (mode.equalsIgnoreCase("B")) {
            goldAppearance.setTexCoordGeneration(texCoordGenLinear);
            texAttr.setTextureMode(TextureAttributes.MODULATE);
        }
        else if (mode.equalsIgnoreCase("C")) {
            goldAppearance.setTexCoordGeneration(null);
            texAttr.setTextureMode(TextureAttributes.MODULATE);
        }
        
        if (shapeSwitch.getWhichChild() != GOLD_SHAPE3D) {
            isShader = false;
            shapeSwitch.setWhichChild(GOLD_SHAPE3D);
        }
    }

    void setWoodTexture(String mode) {
        if (mode.equalsIgnoreCase("A")) {
            attrValueDistTex.setValue(new Integer(0));  // false
        }
        else if (mode.equalsIgnoreCase("B")) {
            attrValueDistTex.setValue(new Integer(1));  // true
        }

        if (isGLSL && shapeSwitch.getWhichChild() != WOOD_SHAPE3D) {
            isShader = true;
            shapeSwitch.setWhichChild(WOOD_SHAPE3D);
        }
    }

    void closeUniverse() {
        view.removeAllCanvas3Ds();
        view.attachViewPlatform(null);
        locale.getVirtualUniverse().removeAllLocales();
    }

    //
    // Create universe
    //
    private void initUniverse() {

        createUniverse();
        createScene();
        
        setLive();

        // Start animation
        distortBeh.startStop(true);
    }
    
    // Setup navigator
    private void setupNavigator(JPanel component) {

        orbitBehInterim.setAWTComponent(component);

        double sceneRadius = 1.0f;

        Bounds bounds = sceneBranch.getBounds();
        BoundingSphere sphereBounds = null;

        if (bounds.isEmpty()) {
            sphereBounds = new BoundingSphere();
        }
        else {
            if (bounds instanceof BoundingSphere)
                sphereBounds = (BoundingSphere)bounds;
            else
                sphereBounds = new BoundingSphere(bounds);
        }

        sceneRadius = sphereBounds.getRadius();

        orbitBehInterim.setTransFactors(sceneRadius/4.0f, sceneRadius/4.0f);
        orbitBehInterim.setZoomFactor(sceneRadius/4.0f);
        orbitBehInterim.setRotFactors(0.5f, 0.5f);

        orbitBehInterim.setProjectionMode(View.PARALLEL_PROJECTION);
        orbitBehInterim.setPureParallelEnabled(true);

        setVantagePoint("Front");
    }

    private void createScene() {

        // String3D Geometry
        
        GeneralPath extrPath = new GeneralPath();

        float extend = 3.0f;
        float depth = 50.0f;
        float cut = depth/10.0f;

        extrPath.moveTo(0.0f, 0.0f);
        extrPath.lineTo(cut, extend);
        extrPath.lineTo(depth-cut, extend);
        extrPath.lineTo(depth, 0.0f);

        AWTShapeExtrusion extrusion = new AWTShapeExtrusion(extrPath); 
        AWTShapeExtruder  extruder  = new AWTShapeExtruder(0.15, extrusion, Math.toRadians(24));

        Font font = new Font("Amble Cn", Font.PLAIN, 100); // Dialog

        String3D string3D = new String3D(font, extruder);
        string3D.setPosition(new Point3f(0, 0, 0)); 
        string3D.setCharacterSpacing(2.0f);
        string3D.setAlignment(String3D.Alignment.CENTER);
        string3D.setPath(String3D.Path.RIGHT);

        Transform3D scaleT3D = new Transform3D();
        scaleT3D.setScale(0.01);                             // scale down 1/100
        scaleT3D.setTranslation(new Vector3d(0, 0, -0.25));  // center z
        extruder.setGeometryTransform(scaleT3D);

        BoundingBox bBox = new BoundingBox();
        
        GeometryArray geom3D = string3D.getStringGeometry("FX3D", bBox);
        
        Point3d lower = new Point3d();
        Point3d upper = new Point3d();
        bBox.getLower(lower);
        bBox.getUpper(upper);

        // Coordinates and normals for BY_REFERENCE
        int vertexCt = geom3D.getVertexCount();
        float[] coords = new float[vertexCt*3];
        float[] normals = new float[vertexCt*3];

        geom3D.getCoordinates(0, coords);
        geom3D.getNormals(0, normals);

        // Texture coordinates for BY_REFERENCE
        double xScale = 1.0/(upper.x-lower.x);
        double yScale = 1.0f/(upper.y-lower.y);
        float planeSx = (float)xScale;
        float planeSw = (float)(-lower.x * xScale);
        float planeTy = (float)yScale;
        float planeTw = (float)(-lower.y * yScale);
        
        Vector4f planeS = new Vector4f(planeSx, 0f, 0f, planeSw);
        Vector4f planeT = new Vector4f(0.0f, planeTy, 0f, planeTw);

        float[] texCoords = new float[vertexCt*2];

        int t = 0;
        for (int i=0; i < vertexCt; i++) {
            texCoords[t++] = coords[i*3]*planeSx + planeSw;
            texCoords[t++] = coords[i*3 + 1]*planeTy + planeTw;
        }

        TriangleArray goldGeom = new TriangleArray(vertexCt,
                                                   GeometryArray.COORDINATES | 
                                                   GeometryArray.NORMALS |
                                                   GeometryArray.TEXTURE_COORDINATE_2 |
                                                   GeometryArray.BY_REFERENCE);
        goldGeom.setCapability(GeometryArray.ALLOW_REF_DATA_WRITE);
        goldGeom.setCoordRefFloat(coords);
        goldGeom.setNormalRefFloat(normals);
        goldGeom.setTexCoordRefFloat(0, texCoords);
                
//System.out.println("vertexCt = " + vertexCt);
//System.out.println("bBox = " + bBox);
//System.out.println("planeSx / planeSw / planeTy / planeTw= " + planeSx + " / " + planeSw + " / " + planeTy + " / " + planeTw);

        // Shape3Ds

        Shape3D goldShape3D = new Shape3D();
        goldShape3D.setAppearance(createGoldAppear(planeS, planeT));
        goldShape3D.setGeometry(goldGeom);

        Shape3D woodShape3D = new Shape3D();
        woodShape3D.setAppearance(createWoodAppear());
        woodShape3D.setGeometry(geom3D);


        // DistortBehavior

        distortBeh = new DistortBehavior(goldGeom);
        distortBeh.setSchedulingBounds(new BoundingSphere());

        // Rotate string due to DistortBehavior's calculations
        TransformGroup rotateTG  = new TransformGroup();
        Transform3D rotateT3D = new Transform3D();
        rotateT3D.setEuler(new Vector3d(0, -Math.toRadians(80), 0));
        rotateTG.setTransform(rotateT3D);

        // Toggle shapes
        shapeSwitch = new Switch(Switch.CHILD_NONE);
        shapeSwitch.setCapability(Switch.ALLOW_SWITCH_WRITE);
        shapeSwitch.addChild(goldShape3D);
        shapeSwitch.addChild(woodShape3D);

        rotateTG.addChild(shapeSwitch);
        
        sceneBranch.addChild(rotateTG);
        sceneBranch.addChild(distortBeh);
    }
    
    private Appearance createGoldAppear(Vector4f planeS, Vector4f planeT) {
        
        goldAppearance = new Appearance();
        goldAppearance.setCapability(Appearance.ALLOW_TEXGEN_WRITE);

        // White diffuse color for modulate texture blending
        Material material = new Material();
        material.setSpecularColor(0.0f, 0.0f, 0.0f);

        goldAppearance.setMaterial(material);

        // Gold texture
        Texture texture = new TextureLoader(this.getClass().getResource("gold.jpg"), null).getTexture();
        texture.setMagFilter(Texture.NICEST);
        texture.setMinFilter(Texture.NICEST);

        goldAppearance.setTexture(texture);

        // TexCoordGenerations: OBJECT_LINEAR | SPHERE_MAP texture coordinates
        texCoordGenLinear = new TexCoordGeneration(TexCoordGeneration.OBJECT_LINEAR, TexCoordGeneration.TEXTURE_COORDINATE_2);
        texCoordGenLinear.setPlaneS(planeS);
        texCoordGenLinear.setPlaneT(planeT);

        texCoordGenSphere = new TexCoordGeneration(TexCoordGeneration.SPHERE_MAP, TexCoordGeneration.TEXTURE_COORDINATE_2);
                
        goldAppearance.setTexCoordGeneration(texCoordGenSphere);    // at start time

        // TextureAttributes: MODULATE | REPLACE blending
        texAttr = new TextureAttributes();
        texAttr.setCapability(TextureAttributes.ALLOW_MODE_WRITE);
        texAttr.setTextureMode(TextureAttributes.REPLACE);          // at start time

        goldAppearance.setTextureAttributes(texAttr);

        return goldAppearance;
    }

    private ShaderAppearance createWoodAppear() {

        String vertexProgram = null;
        String fragmentProgram = null;
        try {
            vertexProgram = StringIO.readFully(this.getClass().getResource("WoodDistort.vert"));
            fragmentProgram = StringIO.readFully(this.getClass().getResource("WoodDistort.frag"));
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }

        Shader[] shaders = new Shader[2];
        shaders[0] = new SourceCodeShader(Shader.SHADING_LANGUAGE_GLSL,
                                          Shader.SHADER_TYPE_VERTEX,
                                          vertexProgram);
        shaders[1] = new SourceCodeShader(Shader.SHADING_LANGUAGE_GLSL,
                                          Shader.SHADER_TYPE_FRAGMENT,
                                          fragmentProgram);
        String[] shaderAttrNames = {
            "Spx",
            "Spy",
            "IsDistortTexture",
            "Scale", 
            "GrainSizeRecip",
            "DarkColor",
            "Spread"
        };
        Object[] shaderAttrValues = {
            new Float(0.0f),
            new Float(0.0f),
            new Integer(0),   // Boolean  0? false : true
            new Float(2.0f), 
            new Float(5.0f), 
            new Color3f(0.6f, 0.3f, 0.1f),
            new Color3f(0.15f, 0.075f, 0.0f)
        };
        ShaderProgram shaderProgram = new GLSLShaderProgram();
        shaderProgram.setShaders(shaders);
        shaderProgram.setShaderAttrNames(shaderAttrNames);

        ShaderAttributeSet shaderAttributeSet = new ShaderAttributeSet();

        attrValueSpx = new ShaderAttributeValue(shaderAttrNames[0],  shaderAttrValues[0]);
        attrValueSpx.setCapability(ShaderAttributeValue.ALLOW_VALUE_WRITE);

        attrValueSpy = new ShaderAttributeValue(shaderAttrNames[1],  shaderAttrValues[1]);
        attrValueSpy.setCapability(ShaderAttributeValue.ALLOW_VALUE_WRITE);

        attrValueDistTex = new ShaderAttributeValue(shaderAttrNames[2],  shaderAttrValues[2]);
        attrValueDistTex.setCapability(ShaderAttributeValue.ALLOW_VALUE_WRITE);

        shaderAttributeSet.put(attrValueSpx);
        shaderAttributeSet.put(attrValueSpy);
        shaderAttributeSet.put(attrValueDistTex);

        for (int i=3; i < shaderAttrNames.length; i++) {
            ShaderAttributeValue shaderAttribute =
                new ShaderAttributeValue(shaderAttrNames[i],  shaderAttrValues[i]);
            shaderAttributeSet.put(shaderAttribute);
        }

        ShaderAppearance shaderAppear = new ShaderAppearance();
        shaderAppear.setShaderAttributeSet(shaderAttributeSet);
        shaderAppear.setShaderProgram(shaderProgram);

        return shaderAppear;
    }

    // Set live
    private void setLive() {
        locale.addBranchGraph(sceneBranch);
        locale.addBranchGraph(viewBranch);
        locale.addBranchGraph(enviBranch);
    }

    // Base world
    private void createUniverse() {

        // Bounds
    	globalBounds = new BoundingSphere();
        globalBounds.setRadius(Double.MAX_VALUE);

        //
        // Viewing
        //
        view = new View();
        view.setPhysicalBody(new PhysicalBody());
        view.setPhysicalEnvironment(new PhysicalEnvironment());
        view.setBackClipPolicy(View.VIRTUAL_EYE);
        view.setFrontClipPolicy(View.VIRTUAL_EYE);
        view.setBackClipDistance(15);
        view.setFrontClipDistance(0.01);

        //
        // SuperStructure
        //
        VirtualUniverse vu = new VirtualUniverse();
        locale = new Locale(vu);

        //
        // BranchGraphs
        //
        sceneBranch = new BranchGroup();
        viewBranch = new BranchGroup();
        enviBranch = new BranchGroup();

        // ViewBranch

        TransformGroup viewTG = new TransformGroup();
        viewTG.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);

        ViewPlatform vp = new ViewPlatform();
        view.attachViewPlatform(vp);

        orbitBehInterim = new OrbitBehaviorInterim(OrbitBehaviorInterim.REVERSE_ALL);
        orbitBehInterim.setViewingTransformGroup(viewTG);
        orbitBehInterim.setVpView(view);
        orbitBehInterim.setSchedulingBounds(globalBounds);
        orbitBehInterim.setClippingEnabled(false); // fix clipping distances

        Transform3D homeTransform = new Transform3D();
        homeTransform.setTranslation(new Vector3d(0.0, 0.45, 4.0));
        orbitBehInterim.setHomeTransform(homeTransform);
        orbitBehInterim.setHomeRotationCenter(new Point3d(0.0, 0.0, 0.0));

        DirectionalLight headLight = new DirectionalLight();
        headLight.setInfluencingBounds(globalBounds);

        viewTG.addChild(vp);
        viewTG.addChild(orbitBehInterim);
        viewTG.addChild(headLight);

        viewBranch.addChild(viewTG);

        // EnviBranch

        Background bg = new Background();
        bg.setApplicationBounds(globalBounds);
        bg.setColor(new Color3f(bgColor));

        enviBranch.addChild(bg);
    }

    // Derived from "org.jdesktop.j3d.examples.distort_glyph.DistortBehavior.java"
    // See license notice above
    private final class DistortBehavior extends Behavior implements GeometryUpdater {

        private WakeupOnBehaviorPost    startCriterion      =   null;
        private WakeupOr                stopFrameCriterion  =   null;

        private final int       START           =   98;
        private final int       STOP            =   99;
    
        private final long      MinDuration     =   10000L;                      // max speed
        private long            newDuration     =   (long)(MinDuration*100f/30); // start speed

        private Alpha           alpha           =   null;
    
        private float           currAlphaValue  =   0.0f;
        private float           lastAlphaValue  =   Float.NEGATIVE_INFINITY;

        private final double    TwoPi3          =   Math.PI*2*3;
        private final double    TwoPi5          =   Math.PI*2*5;

        private float           spx             =   0.0f;
        private float           spy             =   0.0f;

        private GeometryArray	geometryArray   =   null;

        private Vector3f        coord           =   new Vector3f();
        private	float[]         origCoordArray  =   null;
        private	float[]         copyCoordArray  =   null;
        private int             coordsLength    =   0;

        private Vector3f        normal          =   new Vector3f();
        private	float[]         origNormalArray =   null;
        private	float[]         copyNormalArray =   null;

        private Transform3D     t3              =   new Transform3D();

        DistortBehavior(GeometryArray geometry) {

            startCriterion = new WakeupOnBehaviorPost(this, START);
            WakeupCriterion[] criterions = {new WakeupOnBehaviorPost(this, STOP), new WakeupOnElapsedFrames(0)};
            stopFrameCriterion = new WakeupOr(criterions);

            alpha = new Alpha();
            alpha.setIncreasingAlphaDuration(newDuration);
            alpha.setLoopCount(-1);
            alpha.setStartTime(System.currentTimeMillis());
            alpha.pause();

            geometryArray = geometry;

            origCoordArray = geometry.getCoordRefFloat();
            copyCoordArray = (float[])origCoordArray.clone();

            origNormalArray = geometry.getNormalRefFloat();
            copyNormalArray = (float[])origNormalArray.clone();

            coordsLength = origCoordArray.length;
        }

        private void startStop(boolean start) {
            if (start) postId(START);
            else postId(STOP);
        }

        // speed = [1, 100]
        private void speed(int s) {

            // Loop duration determines rotation speed
            // New IncreasingAlphaDuration
            newDuration = (long)( MinDuration * 100f/s );

            // is running
            if (!alpha.isPaused()) {
                alpha.pause(System.currentTimeMillis()); // smoother
                resumeAlpha(newDuration);
            }
        }

        // Resume Alpha with new duration/speed
        private void resumeAlpha(long duration) {

            long oldStartTime = alpha.getStartTime();
            long pauseTime = alpha.getPauseTime();
            float pauseValue = alpha.value();

            // Offset according to alpha's pauseValue and the new IncreasingAlphaDuration
            long resumeOffsetTime = (long)(pauseValue * duration);

            alpha.setIncreasingAlphaDuration(duration);

            // Resume

            // Alpha source code: newStartTime = oldStartTime + resumeTime - pauseTime
            // Start immediately and adapt new duration:
            //   => System.currentTimeMillis - resumeOffsetTime = oldStartTime + resumeTime - pauseTime
            //   => resumeTime = System.currentTimeMillis - resumeOffsetTime - oldStartTime + pauseTime

            alpha.resume(System.currentTimeMillis() - resumeOffsetTime - oldStartTime + pauseTime);
        }

        // Behavior

        @Override
        public void initialize() {
            wakeupOn(startCriterion);
        }

        @Override
        public void processStimulus(Enumeration criteria) {
            while (criteria.hasMoreElements()) {
                WakeupCriterion wakeUp = (WakeupCriterion)criteria.nextElement();
                // 3D string distortion
                if (wakeUp instanceof WakeupOnElapsedFrames) {
                    currAlphaValue = alpha.value();
                    if (currAlphaValue != lastAlphaValue) {
                        lastAlphaValue = currAlphaValue;
                        
                        spx = (float)Math.sin(TwoPi3 * currAlphaValue);
                        spy = (float)Math.cos(TwoPi5 * currAlphaValue);
	
                        if (isShader) {
                            attrValueSpx.setValue(new Float(spx));
                            attrValueSpy.setValue(new Float(spy));
                        }
                        else {
                            geometryArray.updateData(this);
                        }
                    }

                    wakeupOn(stopFrameCriterion);
                }
                // Start/Stop
                else if (wakeUp instanceof WakeupOnBehaviorPost) {
                    int postId = ((WakeupOnBehaviorPost)wakeUp).getTriggeringPostId();
                    if (postId == START) {
                        if (alpha.getIncreasingAlphaDuration() != newDuration)
                            resumeAlpha(newDuration);
                        else
                            alpha.resume();
                    	
                        wakeupOn(stopFrameCriterion);
                    }
                    else if (postId == STOP) {
                        alpha.pause();
                        // Synchronize vertices
                        if (isShader) {
                            geometryArray.updateData(this);
                        }
                        else {
                            attrValueSpx.setValue(new Float(spx));
                            attrValueSpy.setValue(new Float(spy));
                        }

                        wakeupOn(startCriterion);
                        return;
                    }
                }
                else {
                    wakeupOn(startCriterion);
                }
            }
        }

        // Interface GeometryUpdater
        @Override
        public void updateData(Geometry geometry) {

            t3.setIdentity();

            for (int n=0; n < coordsLength; n+=3) {

                // Coordinates

                coord.x = copyCoordArray[n];
                coord.y = copyCoordArray[n+1];
                coord.z = copyCoordArray[n+2];

                float px = coord.x - spx;
                float py = coord.y - spy;
                float pz = coord.z;

                float d = (float) Math.sqrt(px*px + py*py + pz*pz);

                //t3.rotZ(d);
                //t3.rotX(d*2);
                t3.rotY(d);
                t3.transform(coord);

                origCoordArray[n]   = coord.x;
                origCoordArray[n+1] = coord.y;
                origCoordArray[n+2] = coord.z;

                // Normals

                normal.x = copyNormalArray[n];
                normal.y = copyNormalArray[n+1];
                normal.z = copyNormalArray[n+2];

                t3.transform(normal); // t3 is orthogonal

                origNormalArray[n]   = normal.x;
                origNormalArray[n+1] = normal.y;
                origNormalArray[n+2] = normal.z;
            }
        }
    }

    // Java 3D properties
    void printJava3DProps() {

        Map vuProps = VirtualUniverse.getProperties();

        System.out.println("Java 3D Version  =  " + vuProps.get("j3d.version"));
        System.out.println("Java 3D Renderer =  " + vuProps.get("j3d.renderer"));
        System.out.println("Java 3D Pipeline =  " + vuProps.get("j3d.pipeline"));
        System.out.println("------------------------------------------------------------------------");

        if (offCanvas3D != null) {
            Map c3dProps = offCanvas3D.queryProperties();

            System.out.println("Native  Version  =  " + c3dProps.get("native.version"));
            System.out.println("Native  GLSL     =  " + c3dProps.get("shadingLanguageGLSL"));
            System.out.println("Native  Vendor   =  " + c3dProps.get("native.vendor"));
            System.out.println("Native  Renderer =  " + c3dProps.get("native.renderer"));

            System.out.println("------------------------------------------------------------------------");
        }
        System.out.println("");
    }
}
