
package com.interactivemesh.j3d.testspace.jfx.hellocube;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.GradientPaint;
import java.awt.Graphics2D;

import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;

import java.awt.image.BufferedImage;

import java.util.Map;

import javax.media.j3d.Alpha;
import javax.media.j3d.RotationInterpolator;

import javax.media.j3d.Appearance;
import javax.media.j3d.Background;
import javax.media.j3d.BoundingSphere;
import javax.media.j3d.BranchGroup;
import javax.media.j3d.Canvas3D;
import javax.media.j3d.DirectionalLight;
import javax.media.j3d.GeometryArray;
import javax.media.j3d.GraphicsConfigTemplate3D;
import javax.media.j3d.ImageComponent2D;
import javax.media.j3d.IndexedTriangleArray;
import javax.media.j3d.Locale;
import javax.media.j3d.Material;
import javax.media.j3d.PhysicalBody;
import javax.media.j3d.PhysicalEnvironment;
import javax.media.j3d.Shape3D;
import javax.media.j3d.Transform3D;
import javax.media.j3d.TransformGroup;
import javax.media.j3d.View;
import javax.media.j3d.ViewPlatform;
import javax.media.j3d.VirtualUniverse;

import javax.swing.JPanel;

import javax.vecmath.AxisAngle4d;
import javax.vecmath.Color3f;
import javax.vecmath.Vector3d;

// FXCanvas3D API 3.0, see http://www.interactivemesh.org/testspace/j3dmeetsjfx.html
//import com.interactivemesh.j3d.community.gui.FXCanvas3DDB; // Alternative !!
import com.interactivemesh.j3d.community.gui.FXCanvas3DSB;
import com.interactivemesh.j3d.community.gui.FXCanvas3DRepainter;

// JavaFX
import javafx.async.RunnableFuture;

/**
 * HelloCubeUniverse.java
 *
 * Version: 2.0
 * Date: 2010/05/14
 *
 * Copyright (c) 2009-2010
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * but you may not modify, compile, or distribute this 'HelloCubeUniverse.java'.
 *
 */
final class HelloCubeUniverse implements RunnableFuture {

    static {
        System.out.println("\nHelloCubeUniverse : Copyright (c) 2009-2010 August Lammersdorf, www.InteractiveMesh.com.");
    }

    private BoundingSphere      globalBounds        = 	null;

    private View                view                = 	null;
    private Locale              locale              = 	null;
    private Canvas3D            offCanvas3D         =   null;

    private BranchGroup         sceneBranch         =   null;
    private BranchGroup         viewBranch          = 	null;
    private BranchGroup         enviBranch          = 	null;

    private Background          background          =   null;
    
    private Color               bgColor0            =   new Color(0, 102, 204); // RGB 0.0f, 0.4f, 0.8f, HSB 209, 100,  80
    private Color               bgColor1            =   new Color(0, 153, 255); // RGB 0.0f, 0.6f, 1.0f, HSB 203, 100, 100

    
    // Constructor: done in method run().
    HelloCubeUniverse() {
    }
    
    // JavaFX Interface RunnableFuture
    // Called from a thread other than the JavaFX event dispatch thread.
    public void run() {
        initUniverse();
    }

    // Creates and returns the lightweight 3D canvas
    // Alternative: FXCanvas3DDB
    FXCanvas3DSB createFXCanvas3D(FXCanvas3DRepainter repainter, JPanel parent) {

        FXCanvas3DSB fxCanvas3D = null;

        try {
            GraphicsConfigTemplate3D gCT = new GraphicsConfigTemplate3D();
            
            fxCanvas3D = new FXCanvas3DSB(gCT);
            fxCanvas3D.setRepainter(repainter);
        }
        catch (Exception e) {
            System.out.println("HelloCubeUniverse: FXCanvas3D failed !!");
            e.printStackTrace();
            System.exit(0);
        }

        // Optional
        createBackgroundImage(fxCanvas3D);
        // Optional: less 'flickering' during resizing
        fxCanvas3D.setBackground(bgColor0);

        // Due to Java 3D's implementation of off-screen rendering:
        // 1. Set size
        // 2. Provide a parent
        // 3. Get the heavyweight off-screen Canvas3D and add this to the view object

        // 1. Size
        Dimension dim = parent.getSize();
        if (dim.width < 1 || dim.height < 1) {
            dim.width = 100;
            dim.height = 100;
            parent.setSize(dim);
        }
        parent.setPreferredSize(dim);
        fxCanvas3D.setPreferredSize(dim);
        fxCanvas3D.setSize(dim);

        // 2. Parent
        // BorderLayout
        parent.setLayout(new BorderLayout());
        parent.add(fxCanvas3D, BorderLayout.CENTER);

        // 3. Heavyweight off-screen Canvas3D of the lightweight FXCanvas3DDB/SB
        offCanvas3D = fxCanvas3D.getOffscreenCanvas3D();
        if (offCanvas3D != null) {
            // View renders into the off-screen Canvas3D
            view.addCanvas3D(offCanvas3D);
        }
        else {
            System.out.println("HelloCubeUniverse: Off-screen Canvas3D is null !!");
            System.exit(0);
        }
        
        return fxCanvas3D;
    }

    // Shut down
    void closeUniverse() {
        view.removeAllCanvas3Ds();
        view.attachViewPlatform(null);
        locale.getVirtualUniverse().removeAllLocales();
    }

    //
    // Init universe 
    //
    private void initUniverse() {

        createUniverse();
        createScene();

        setLive();
    }

    private void createScene() {

        TransformGroup targetTG = new TransformGroup();
        targetTG.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);

        Alpha alpha = new Alpha();
        alpha.setLoopCount(-1);
        alpha.setIncreasingAlphaDuration(6000);
        alpha.setStartTime(System.currentTimeMillis());

        RotationInterpolator rotInt = new RotationInterpolator(alpha, targetTG);
        rotInt.setSchedulingBounds(globalBounds);


        // Cube model

        Shape3D cubeShape = buildCubeShape();
        
        // Cube world position
        
        TransformGroup cubeTG = new TransformGroup();
        Transform3D cubeTransform = new Transform3D();
        cubeTransform.setRotation(new AxisAngle4d(0.0, 1.0, 0.0, -0.52358));
        cubeTG.setTransform(cubeTransform);
        
        // Scene graph
        
        cubeTG.addChild(cubeShape);

        targetTG.addChild(rotInt);
        targetTG.addChild(cubeTG);
        
        sceneBranch.addChild(targetTG);
    }
    
    private Shape3D buildCubeShape() {
      
        // Appearances of cube 
        
        Appearance cubeAppear = new Appearance();

        Material cubeMaterial = new Material();
        cubeMaterial.setAmbientColor(0.0f, 0.0f, 0.0f);
        cubeMaterial.setDiffuseColor(new Color3f(0.8f, 0.0f, 0.0f));
        cubeMaterial.setEmissiveColor(new Color3f(0.2f, 0.0f, 0.0f));
        cubeMaterial.setSpecularColor(0.0f, 0.0f, 0.0f);
        cubeMaterial.setShininess(64f);

        cubeAppear.setMaterial(cubeMaterial);

        // Geometry of cube, length of edges = 1.0
        
        // Points
        float[] coords = {
            -0.5f,  0.5f, -0.5f,    // 0: left  top  back
            -0.5f,  0.5f,  0.5f,    // 1: left  top  front
             0.5f,  0.5f,  0.5f,    // 2: right top  front
             0.5f,  0.5f, -0.5f,    // 3: right top  back
            -0.5f, -0.5f,  0.5f,    // 4: left  bot  front
             0.5f, -0.5f,  0.5f,    // 5: right bot  front
             0.5f, -0.5f, -0.5f,    // 6: right bot  back
            -0.5f, -0.5f, -0.5f     // 7: left  bot  back
        };
        
        // Normals
        float[] normals = {
             0.0f,  0.0f,  1.0f,    // 0: front
             0.0f,  0.0f, -1.0f,    // 1: back
            -1.0f,  0.0f,  0.0f,    // 2: left
             1.0f,  0.0f,  0.0f,    // 3: right
             0.0f,  1.0f,  0.0f,    // 4: top
             0.0f, -1.0f,  0.0f     // 5: bottom
        };
        
        // Triangles
        int[] indicesC = {
            0, 1, 2,    // top
            0, 2, 3,    // top
            1, 4, 5,    // front
            1, 5, 2,    // front
            2, 5, 6,    // right
            2, 6, 3,    // right
            3, 6, 7,    // back
            3, 7, 0,    // back
            0, 7, 4,    // left
            0, 4, 1,    // left
            5, 4, 7,    // bottom
            5, 7, 6     // bottom
        };

        int[] indicesN = {
            4, 4, 4,    // top
            4, 4, 4,    // top
            0, 0, 0,    // front
            0, 0, 0,    // front
            3, 3, 3,    // right
            3, 3, 3,    // right
            1, 1, 1,    // back
            1, 1, 1,    // back
            2, 2, 2,    // left
            2, 2, 2,    // left
            5, 5, 5,    // bottom
            5, 5, 5     // bottom
        };
        
        IndexedTriangleArray cubeGeometry = new IndexedTriangleArray(
            8, // max of coordinates and normals,
            GeometryArray.COORDINATES | GeometryArray.NORMALS,
            36);
        
        cubeGeometry.setCoordinates(0, coords);
        cubeGeometry.setNormals(0, normals);
        cubeGeometry.setCoordinateIndices(0, indicesC);
        cubeGeometry.setNormalIndices(0, indicesN);
      
        Shape3D cubeShape = new Shape3D(cubeGeometry, cubeAppear);
        
        return cubeShape;
    }

    // Set live
    private void setLive() {
	sceneBranch.compile();
        locale.addBranchGraph(sceneBranch);
        locale.addBranchGraph(viewBranch);
        locale.addBranchGraph(enviBranch);
    }

    // Base world
    private void createUniverse() {

        // Bounds
    	globalBounds = new BoundingSphere();
        globalBounds.setRadius(Double.MAX_VALUE);

        //
        // Viewing
        //
        view = new View();
        view.setPhysicalBody(new PhysicalBody());
        view.setPhysicalEnvironment(new PhysicalEnvironment());

        //
        // SuperStructure
        //
        VirtualUniverse vu = new VirtualUniverse();
        locale = new Locale(vu);

        //
        // BranchGraphs
        //
        sceneBranch = new BranchGroup();
        viewBranch = new BranchGroup();
        enviBranch = new BranchGroup();

        // ViewBranch

        TransformGroup viewTG = new TransformGroup();
        Transform3D viewTransform = new Transform3D();
        viewTransform.setRotation(new AxisAngle4d(1.0, 0.0, 0.0, -0.52358));
        viewTransform.setTranslation(new Vector3d(0.0, 1.8, 3.25));
        viewTG.setTransform(viewTransform);

        ViewPlatform vp = new ViewPlatform();
        view.attachViewPlatform(vp);

        DirectionalLight headLight = new DirectionalLight();
        headLight.setInfluencingBounds(globalBounds);

        viewTG.addChild(vp);
        viewTG.addChild(headLight);

        viewBranch.addChild(viewTG);

        // EnviBranch

        background = new Background();
        background.setCapability(Background.ALLOW_IMAGE_WRITE);
        background.setApplicationBounds(globalBounds);
        background.setColor(new Color3f(bgColor0));

        enviBranch.addChild(background);
    }

    // Image for Background node
    private void createBackgroundImage(final JPanel canvas3D) {
        canvas3D.addComponentListener(new ComponentAdapter() {
            @Override
            public void componentResized(ComponentEvent e) {
                int width = canvas3D.getWidth();
                int height = canvas3D.getHeight();

                ImageComponent2D currImage = background.getImage();
                if (currImage != null) {
                    if (currImage.getWidth() == width && currImage.getHeight() == height)
                        return;
                }
               
                BufferedImage gradientImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
                Graphics2D g2d = (Graphics2D)gradientImage.getGraphics();

                int splitHeight = (int)(height/5f); // yUp !

                GradientPaint gp = new GradientPaint(0, 0,           bgColor0,
                                                     0, splitHeight, bgColor1);
                g2d.setPaint(gp);
                g2d.fillRect(0, 0, width, splitHeight);

                gp = new GradientPaint(0, splitHeight, bgColor1,
                                       0, height,      bgColor0);
                g2d.setPaint(gp);
                g2d.fillRect(0, splitHeight, width, height);

                g2d.dispose();

                background.setImage(new ImageComponent2D(ImageComponent2D.FORMAT_RGB, gradientImage, true, true));
            }
        });
    }

    // Java 3D properties
    void printJava3DProps() {

        Map vuProps = VirtualUniverse.getProperties();

        System.out.println("Java 3D Version  =  " + vuProps.get("j3d.version"));
        System.out.println("Java 3D Renderer =  " + vuProps.get("j3d.renderer"));
        System.out.println("Java 3D Pipeline =  " + vuProps.get("j3d.pipeline"));
        System.out.println("------------------------------------------------------------------------");

        if (offCanvas3D != null) {
            Map c3dProps = offCanvas3D.queryProperties();

            System.out.println("Native  Version  =  " + c3dProps.get("native.version"));
            System.out.println("Native  GLSL     =  " + c3dProps.get("shadingLanguageGLSL"));
            System.out.println("Native  Vendor   =  " + c3dProps.get("native.vendor"));
            System.out.println("Native  Renderer =  " + c3dProps.get("native.renderer"));

            System.out.println("------------------------------------------------------------------------");
        }
        System.out.println("");
    }
}
