package com.interactivemesh.j3d.testspace.orbitbehavior;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Toolkit;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.media.j3d.Appearance;
import javax.media.j3d.Background;
import javax.media.j3d.BoundingBox;
import javax.media.j3d.BoundingSphere;
import javax.media.j3d.BranchGroup;
import javax.media.j3d.Canvas3D;
import javax.media.j3d.DirectionalLight;
import javax.media.j3d.Font3D;
import javax.media.j3d.FontExtrusion;
import javax.media.j3d.GraphicsConfigTemplate3D;
import javax.media.j3d.Locale;
import javax.media.j3d.Material;
import javax.media.j3d.Node;
import javax.media.j3d.PhysicalBody;
import javax.media.j3d.PhysicalEnvironment;
import javax.media.j3d.PickInfo;
import javax.media.j3d.Shape3D;
import javax.media.j3d.Transform3D;
import javax.media.j3d.TransformGroup;
import javax.media.j3d.TriangleArray;
import javax.media.j3d.View;
import javax.media.j3d.ViewPlatform;
import javax.media.j3d.VirtualUniverse;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.SwingUtilities;
import javax.swing.border.TitledBorder;

import javax.vecmath.AxisAngle4f;
import javax.vecmath.Color3f;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;
import javax.vecmath.Vector3f;

import com.sun.j3d.exp.swing.JCanvas3D;
import com.sun.j3d.utils.pickfast.PickCanvas;

// Version 1.1
import com.interactivemesh.j3d.community.utils.navigation.orbit.OrbitBehaviorInterim;

/**
 * This sample program 'JCanvas3DNavigation' is built on a standard VirtualUniverse, 
 * renders into a JCanvas3D, provides navigation by OrbitBehaviorInterim (1.1), 
 * and allows picking for rotation center by a double clicked left mouse button 
 * based on the utility PickCanvas. 
 *
 * PickCanvas seems to work well even if a off-screen Canvas3D (of JCanvas3D) is used. 
 * 
 * @author Copyright (c) 2007 August Lammersdorf, www.InteractiveMesh.com
 * @since June 26, 2007
 * @version 1.1
 * 
 * Please create your own implementation.
 * You are allowed to copy all lines you like of this source code, 
 * but you may not modify, compile, or distribute this 'JCanvas3DNavigation'. 
 * 
 * 
 */
public final class JCanvas3DNavigation {
	
	static {
		System.out.println("JCanvas3DNavigation 1.1: Copyright (c) 2007 August Lammersdorf, www.InteractiveMesh.com.");
	}
	
	private BoundingSphere 		globalBounds 		= 	null;

	private View 				view 				= 	null;
	private	JCanvas3D 			jCanvas3D 			=	null; 
	private Canvas3D			offCanvas3D			=	null;
	private Locale 				locale 				= 	null;
	
	private OrbitBehaviorInterim orbitBehInterim	= 	null;
	private boolean  	 		isHomeRotCenter		= 	false;
	private boolean  	 		isLookAtRotCenter	= 	false;
	private boolean  	 		isPickVertex		= 	true;
	
	private BranchGroup 		sceneBranch 		= 	null;
	private BranchGroup 		viewBranch 			= 	null;
	private BranchGroup 		enviBranch 			= 	null;
	
	private TransformGroup 		scaleTG 			= 	null;
	private TransformGroup 		characterTG			= 	null;
		
	private BoundingBox 		glyphBox 			= 	new BoundingBox();
	private PickCanvas 			pickCanvas 			= 	null;
	
	private	Color  				bgColor 			= 	new Color(0.05f, 0.05f, 0.5f);

	
	
	public static void main(String[] args) {
		new JCanvas3DNavigation();
	}

	public JCanvas3DNavigation() {
		createUniverse();
		createScene();
		setLive();
		
		orbitBehInterim.goHome(true);
		
		showJCanvas3D();
	}
		
	private void createScene() {
		
		//
	    // Font3D 
    	//
        Font font = new Font("Dialog", Font.PLAIN, 1);
        Font3D font3D = new Font3D(font, 0.005, new FontExtrusion());
		//
		// Appearances
		//
		Appearance appearOrb = new Appearance();	
		Appearance appearBeh = new Appearance();
		Appearance appearInt = new Appearance();
		
		Material matOrb = new Material();
		matOrb.setDiffuseColor(1.0f, 0.0f, 0.2f);
		matOrb.setSpecularColor(1.0f, 0.0f, 0.0f);
		matOrb.setShininess(64f);
		
		Material matBeh = new Material();
		matBeh.setDiffuseColor(0.0f, 1.0f, 0.2f);
		matBeh.setSpecularColor(0.0f, 1.0f, 0.0f);
		matBeh.setShininess(64f);

		Material matInt = new Material();
		matInt.setDiffuseColor(1.0f, 1.0f, 0.2f);
		matInt.setSpecularColor(1.0f, 1.0f, 0.0f);
		matInt.setShininess(64f);
		
		appearOrb.setMaterial(matOrb);
		appearBeh.setMaterial(matBeh);
		appearInt.setMaterial(matInt);
		
		//
		// Shapes
		//
		//                   O  r   b  i   t   B  e   h   a  v   i   o   r   I  n   t   e   r   i   m
		int[] glyphCodes  = {79,114,98,105,116,66,101,104,97,118,105,111,114,73,110,116,101,114,105,109};
		
		// Save box of this glyph for the next one
		glyphBox = new BoundingBox(new Point3d(0, 0, 0), new Point3d(0, 0, 0));
		TransformGroup lastGlyphTG = createGlyph(glyphCodes[0], font3D, appearOrb);
		characterTG.addChild(lastGlyphTG);
		
		Appearance appear = appearOrb;
		TransformGroup nextGlyphTG = null;
		for (int i=1,z=glyphCodes.length; i < z; i++) {
			
			if (i > 4 && i < 13)
				appear = appearBeh;
			else if (i > 12)
				appear = appearInt;
			
			nextGlyphTG = createGlyph(glyphCodes[i], font3D, appear);
			lastGlyphTG.addChild(nextGlyphTG);
			lastGlyphTG = nextGlyphTG;
		}
	}
	
	private TransformGroup createGlyph(int glyphCode, Font3D font3D, Appearance appear) {
		
		char[] chars = Character.toChars(glyphCode);
		TriangleArray triangleArray = (TriangleArray)font3D.getGlyphGeometry(chars[0]);		
		Shape3D glyphShape = new Shape3D(triangleArray, appear);
		
		TransformGroup glyphTG = new TransformGroup();
		glyphTG.addChild(glyphShape);
				
		Point3d lower = new Point3d();
		Point3d upper = new Point3d();
		glyphBox.getLower(lower);
		glyphBox.getUpper(upper);
		
		// Position relative to the left glyph
		Transform3D glyphT3D = new Transform3D();
		glyphT3D.setTranslation(new Vector3d(upper.x - lower.x, 0, 0));
		glyphTG.setTransform(glyphT3D);
		
		// Save box of this glyph for the next one
		glyphBox = (BoundingBox)glyphShape.getBounds();
		
		return glyphTG;
	}

	private void changeRotationCenter(MouseEvent event) {
				
	    pickCanvas.setShapeLocation(event.getX(), event.getY());
	    PickInfo pickInfo = pickCanvas.pickClosest();
	     
	    if (pickInfo != null) {
	    	 
	    	Node pickedNode = pickInfo.getNode();

	    	Transform3D locToVWord = new Transform3D();
	    	pickedNode.getLocalToVworld(locToVWord);

	    	Point3d rotationPoint = new Point3d();
	    	
	    	if (isPickVertex) {
	    		rotationPoint = pickInfo.getClosestIntersectionPoint();	    	 		    	 
	    	}
	    	else {
	    		BoundingBox pickedBox = (BoundingBox)pickedNode.getBounds();
	    	 
	    		Point3d lower = new Point3d();
	    		Point3d upper = new Point3d();
	    		pickedBox.getLower(lower);
	    		pickedBox.getUpper(upper);
	    		 
	    		rotationPoint.set((upper.x - lower.x)/2, (upper.y - lower.y)/2, (upper.z - lower.z)/2);
	    	}
	    	 
	    	locToVWord.transform(rotationPoint);
	    	orbitBehInterim.setRotationCenter(rotationPoint, isLookAtRotCenter);	
	    }
	}
	
    // Set live
	private void setLive() {
		sceneBranch.compile();
		viewBranch.compile();
		enviBranch.compile();
        locale.addBranchGraph(sceneBranch);
        locale.addBranchGraph(viewBranch);
        locale.addBranchGraph(enviBranch);
	}
	
	// Base world
    private void createUniverse() {		
        // Bounds
    	globalBounds = new BoundingSphere();
        globalBounds.setRadius(Double.MAX_VALUE);		
        //
        // Viewing
        //
        view = new View();
        view.setPhysicalBody(new PhysicalBody());
        view.setPhysicalEnvironment(new PhysicalEnvironment());
        
        GraphicsConfigTemplate3D gCT = new GraphicsConfigTemplate3D();        
        try {
        	jCanvas3D = new JCanvas3D(gCT);
        	jCanvas3D.setResizeMode(JCanvas3D.RESIZE_IMMEDIATELY);
        	
        	jCanvas3D.addMouseListener(new MouseAdapter() {
        		public void mouseClicked(MouseEvent event) {
        			int clicks = event.getClickCount();
        			if (clicks == 2 && SwingUtilities.isLeftMouseButton(event)) {
	        			changeRotationCenter(event);
        			}
        		}
        	});
        }
        catch (Exception e) {
            System.out.println("JCanvas3DNavigation: JCanvas3D failed !!");
            e.printStackTrace();
            System.exit(0);
        }
        
        //
        // SuperStructure
        //
        VirtualUniverse vu = new VirtualUniverse();
        locale = new Locale(vu);        
        //
        // BranchGraphs
        //		
        sceneBranch = new BranchGroup();
        viewBranch = new BranchGroup();
        enviBranch = new BranchGroup();
        
        // ViewBranch
        
        TransformGroup viewTG = new TransformGroup();
        viewTG.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
                       
        ViewPlatform vp = new ViewPlatform();
        view.attachViewPlatform(vp);
        
        orbitBehInterim = new OrbitBehaviorInterim(jCanvas3D, viewTG, OrbitBehaviorInterim.REVERSE_ALL);
        orbitBehInterim.setSchedulingBounds(globalBounds);
        
        Transform3D homeTransform = new Transform3D();
        homeTransform.setTranslation(new Vector3f(0.0f, 0.5f, 15.0f));
        orbitBehInterim.setHomeTransform(homeTransform);        
        orbitBehInterim.setHomeRotationCenter(new Point3d(0.0, 0.0, 2.0));
        
        DirectionalLight headLight = new DirectionalLight();
        headLight.setInfluencingBounds(globalBounds);

        viewTG.addChild(vp);
        viewTG.addChild(orbitBehInterim);
        viewTG.addChild(headLight);
        
        viewBranch.addChild(viewTG);
        
        // EnviBranch
        
        Background bg = new Background();
        bg.setApplicationBounds(globalBounds);
        bg.setColor(new Color3f(bgColor));
        
        enviBranch.addChild(bg);

        // SceneBranch
        
        scaleTG = new TransformGroup();
        scaleTG.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
        
        sceneBranch.addChild(scaleTG);
        
        characterTG = new TransformGroup();        
        // Center of character 
        Transform3D charTransform = new Transform3D();
        // Scale to a depth of 2.0f (0.2f * 10.0)
        charTransform.setScale(new Vector3d(1.0f, 1.0f, 10.0f));
        // Rotate around X
        charTransform.setRotation(new AxisAngle4f(1.0f, 0.0f, 0.0f, -0.35f));
        // Center text
        charTransform.setTranslation(new Vector3f(-4.2f, -0.36f, 0.0f));
        characterTG.setTransform(charTransform);

        scaleTG.addChild(characterTG);        
    }
    
	private void showJCanvas3D() {
		
		Toolkit toolkit = Toolkit.getDefaultToolkit();
        Dimension screenDim = toolkit.getScreenSize();
	    	
        Dimension dim = new Dimension(screenDim.width - 20, screenDim.width / 3);
        jCanvas3D.setPreferredSize(dim);
        jCanvas3D.setSize(dim);
        
        JPanel jPanel = new JPanel(new BorderLayout());
		jPanel.add(jCanvas3D, BorderLayout.CENTER);
		
		// Off-screen Canvas3D of this lightweight JCanvas3D
        offCanvas3D = jCanvas3D.getOffscreenCanvas3D();
        if (offCanvas3D != null) {
        	// View renders into the off-screen Canvas3D
        	view.addCanvas3D(offCanvas3D);
        	
        	// PickCanvas operates on the not visible off-screen Canvas3D
    	    pickCanvas = new PickCanvas(offCanvas3D, sceneBranch);
    	    pickCanvas.setMode(PickInfo.PICK_GEOMETRY); 
    	    pickCanvas.setFlags(PickInfo.NODE | PickInfo.CLOSEST_INTERSECTION_POINT); 
    	    pickCanvas.setTolerance(4.0f);
        }
        else {
            System.out.println("JCanvas3DNavigation: OffscreenCanvas3D = null !!");
            System.exit(0);
        }

        // GUI / User actions
        
        int fontSize = 14; 
    	if (screenDim.height < 1024)
    		fontSize = 11;
    	else if (screenDim.height < 1200)
        	fontSize = 12;
       
        Font font = new Font("SansSerif", Font.BOLD, fontSize);
        
	    JPanel jPanelLine = new JPanel(new BorderLayout());
	    jPanelLine.setBackground(new Color(0.05f, 0.05f, 0.8f));	    
	    jPanelLine.setBorder(BorderFactory.createEmptyBorder(3, 0, 0, 0));


	    JPanel jPanelActionX = new JPanel();
	    BoxLayout boxLayoutX = new BoxLayout(jPanelActionX, BoxLayout.X_AXIS);
	    jPanelActionX.setLayout(boxLayoutX);
	    jPanelActionX.setBackground(bgColor);	    
	    jPanelActionX.setBorder(BorderFactory.createEmptyBorder(10, 0, 10, 0));
	    
	    jPanelLine.add(jPanelActionX, BorderLayout.CENTER);
	    
	    JPanel jPanelActionXX = new JPanel();
	    BoxLayout boxLayoutXX = new BoxLayout(jPanelActionXX, BoxLayout.X_AXIS);
	    jPanelActionXX.setLayout(boxLayoutXX);
	    jPanelActionXX.setBackground(bgColor);
	    jPanelActionXX.setBorder(BorderFactory.createCompoundBorder(
	    				BorderFactory.createLoweredBevelBorder(), 
	    				BorderFactory.createEmptyBorder(10, 10, 10, 10)));
	    	    	   
	    jPanelActionX.add(Box.createHorizontalGlue());
	    jPanelActionX.add(jPanelActionXX);

	    // Home Transform
	    JPanel jPanelActionHome = new JPanel();
	    BoxLayout boxLayoutHome = new BoxLayout(jPanelActionHome, BoxLayout.Y_AXIS);
	    jPanelActionHome.setLayout(boxLayoutHome);

	    TitledBorder tBorderHome = BorderFactory.createTitledBorder(" Home Transform ");
	    tBorderHome.setTitleFont(font);
	    tBorderHome.setTitleColor(bgColor);
	    tBorderHome.setBorder(BorderFactory.createLineBorder(bgColor));
	    
	    jPanelActionHome.setBorder(BorderFactory.createCompoundBorder(
		    				BorderFactory.createRaisedBevelBorder(), tBorderHome));
	    
	    JButton jButtonHomeTransform = new JButton("Go Home");
	    jButtonHomeTransform.setFont(font);
	    jButtonHomeTransform.setForeground(bgColor);
	    jButtonHomeTransform.setAlignmentX(Component.CENTER_ALIGNMENT);
	    
	    jButtonHomeTransform.addActionListener(new ActionListener() {
	    	public void actionPerformed(ActionEvent event) {
	    		
	    		orbitBehInterim.goHome(isHomeRotCenter);
	    		
	    	}
	    });
	    
	    JCheckBox jChecHomeRotCenter = new JCheckBox("Home Rotation Center");
	    jChecHomeRotCenter.setFont(font);
	    jChecHomeRotCenter.setForeground(bgColor);
	    jChecHomeRotCenter.setAlignmentX(Component.CENTER_ALIGNMENT);
	    
	    jChecHomeRotCenter.addActionListener(new ActionListener() {
	    	public void actionPerformed(ActionEvent event) {
	    		
	    		isHomeRotCenter = ((JCheckBox)event.getSource()).isSelected();
	    		
	    	}
	    });
	    
	    jPanelActionHome.add(jButtonHomeTransform);
	    jPanelActionHome.add(jChecHomeRotCenter);
	    
	    jPanelActionXX.add(jPanelActionHome);	    
	    jPanelActionXX.add(Box.createHorizontalStrut(10));	
	    
	    // Center picking
	    JPanel jPanelActionCenter = new JPanel();
	    BoxLayout boxLayoutCenter = new BoxLayout(jPanelActionCenter, BoxLayout.Y_AXIS);
	    jPanelActionCenter.setLayout(boxLayoutCenter);

	    TitledBorder tBorderCenter = BorderFactory.createTitledBorder(" Rotation Center Picking ");
	    tBorderCenter.setTitleFont(font);
	    tBorderCenter.setTitleColor(bgColor);
	    tBorderCenter.setBorder(BorderFactory.createLineBorder(bgColor));
	    
	    jPanelActionCenter.setBorder(BorderFactory.createCompoundBorder(
		    				BorderFactory.createRaisedBevelBorder(), tBorderCenter));
	    
	    JPanel jPanelPickMode = new JPanel();
	    BoxLayout boxLayoutPickMode = new BoxLayout(jPanelPickMode, BoxLayout.X_AXIS);
	    jPanelPickMode.setLayout(boxLayoutPickMode);
	    jPanelPickMode.setAlignmentX(Component.CENTER_ALIGNMENT);

	    final JRadioButton jRadioPickVertex = new JRadioButton("Pick Vertex");
	    final JRadioButton jRadioPickShape = new JRadioButton("Pick Shape");
	    
	    jRadioPickVertex.setFont(font);
	    jRadioPickShape.setFont(font);
	    
	    jRadioPickVertex.setForeground(bgColor);
	    jRadioPickShape.setForeground(bgColor);
	    
	    ButtonGroup bGroup = new ButtonGroup();
	    bGroup.add(jRadioPickVertex);
	    bGroup.add(jRadioPickShape);
	    
	    jRadioPickVertex.setSelected(true);
	    
	    ItemListener pickModeListener = new ItemListener() {
	    	public void itemStateChanged(ItemEvent event) {
	    		
	    		isPickVertex = jRadioPickVertex.isSelected();
	    		
	    	}
	    };
	    jRadioPickVertex.addItemListener(pickModeListener);
	    jRadioPickShape.addItemListener(pickModeListener);
	    
	    jPanelPickMode.add(jRadioPickVertex);
	    jPanelPickMode.add(Box.createHorizontalStrut(5));
	    jPanelPickMode.add(jRadioPickShape);
	    
	    JCheckBox jCheckLookAtCenter = new JCheckBox("Look at Center");
	    jCheckLookAtCenter.setFont(font);
	    jCheckLookAtCenter.setForeground(bgColor);
	    jCheckLookAtCenter.setAlignmentX(Component.CENTER_ALIGNMENT);
	    
	    jCheckLookAtCenter.addActionListener(new ActionListener() {
	    	public void actionPerformed(ActionEvent event) {
	    		
	    		isLookAtRotCenter = ((JCheckBox)event.getSource()).isSelected();
	    		
	    	}
	    });
	    
	    jPanelActionCenter.add(jPanelPickMode);
	    jPanelActionCenter.add(jCheckLookAtCenter);
	    
	    jPanelActionXX.add(jPanelActionCenter);	   
	    jPanelActionXX.add(Box.createHorizontalStrut(10));
	    
	    // Look At Current Center
	    JPanel jPanelActionLookAt = new JPanel();
	    BoxLayout boxLayoutLookAt = new BoxLayout(jPanelActionLookAt, BoxLayout.Y_AXIS);
	    jPanelActionLookAt.setLayout(boxLayoutLookAt);

	    TitledBorder tBorderLookAt = BorderFactory.createTitledBorder(" Center Transform ");
	    tBorderLookAt.setTitleFont(font);
	    tBorderLookAt.setTitleColor(bgColor);
	    tBorderLookAt.setBorder(BorderFactory.createCompoundBorder(
					    		BorderFactory.createLineBorder(bgColor),
					    		BorderFactory.createEmptyBorder(0, 4, 0, 4)));
	    
	    jPanelActionLookAt.setBorder(BorderFactory.createCompoundBorder(
		    				BorderFactory.createRaisedBevelBorder(), tBorderLookAt));
	    
	    JButton jButtonLookAt = new JButton("Look at Center");
	    jButtonLookAt.setFont(font);
	    jButtonLookAt.setForeground(bgColor);
	    jButtonLookAt.setAlignmentX(Component.CENTER_ALIGNMENT);
	    
	    jButtonLookAt.addActionListener(new ActionListener() {
	    	public void actionPerformed(ActionEvent event) {
	    		
	    		orbitBehInterim.lookAtRotationCenter();
	    		
	    	}
	    });
	    
	    JPanel spacePanel = new JPanel();
	    spacePanel.setMaximumSize(jButtonLookAt.getPreferredSize());
	    spacePanel.setAlignmentX(Component.CENTER_ALIGNMENT);
	    
	    jPanelActionLookAt.add(jButtonLookAt);
	    jPanelActionLookAt.add(spacePanel);
	    
	    jPanelActionXX.add(jPanelActionLookAt);	    
	    
	    jPanelActionX.add(Box.createHorizontalGlue());

	    jPanel.add(jPanelLine, BorderLayout.SOUTH);
	    
	    // JFrame
	    
		JFrame jFrame = new JFrame();
		jFrame.setTitle("InteractiveMesh : JCanvas3D Navigation");
		jFrame.setDefaultCloseOperation(javax.swing.WindowConstants.EXIT_ON_CLOSE);
		jFrame.add(jPanel);
		jFrame.pack();
		
        Dimension jframeDim = jFrame.getSize();
        jFrame.setLocation((screenDim.width - jframeDim.width)/2, (screenDim.height - jframeDim.height)/2);

		jFrame.setVisible(true);
	}	
}