package org.interactivemesh.pivot.testspace.charcube;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Toolkit;

import org.apache.pivot.collections.Map;

import org.apache.pivot.wtk.Action;
import org.apache.pivot.wtk.Application;
import org.apache.pivot.wtk.Border;
import org.apache.pivot.wtk.BoxPane;
import org.apache.pivot.wtk.Button;
import org.apache.pivot.wtk.ButtonPressListener;
import org.apache.pivot.wtk.ButtonGroup;
import org.apache.pivot.wtk.Component;
import org.apache.pivot.wtk.ComponentListener;
import org.apache.pivot.wtk.DesktopApplicationContext;
import org.apache.pivot.wtk.Display;
import org.apache.pivot.wtk.Expander;
import org.apache.pivot.wtk.HorizontalAlignment;
import org.apache.pivot.wtk.Label;
import org.apache.pivot.wtk.Orientation;
import org.apache.pivot.wtk.Panel;
import org.apache.pivot.wtk.PushButton;
import org.apache.pivot.wtk.RadioButton;
import org.apache.pivot.wtk.Slider;
import org.apache.pivot.wtk.SliderValueListener;
import org.apache.pivot.wtk.SplitPane;
import org.apache.pivot.wtk.Theme;
import org.apache.pivot.wtk.VerticalAlignment;
import org.apache.pivot.wtk.Window;

import org.apache.pivot.wtk.skin.terra.TerraExpanderSkin;

import org.interactivemesh.pivot.wtk.j3d.PCanvas3D;

/*
 * CharacterCube.java
 *
 * Version: 2.0
 * Date: 2011/01/15
 *
 * Copyright (c) 2009 - 2011
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * but you may not modify, compile, or distribute this 'CharacterCube.java'.
 *
 */

public class CharacterCube implements Application { 
    
    private Window window = null;
    
    private CharacterCubeUniverse universe = null;
    
    private Label fpsValueLable = null;
    
    private Font textFont = null;
    private Font titleFont = null;
    
    private int border = 50;
    private int charSize = 16;
    private int titleSize = 20;
    
    private Color bgAction = new Color(0, 153, 255);
    private Color bgPanel = new Color(0, 102, 204);
    private Color bgSplit = new Color(0, 76, 204);
    
    private static Dimension screenSize = null;
    
    // Desktop application only
    public static void main(String[] args) {        
        
        screenSize = Toolkit.getDefaultToolkit().getScreenSize();
        
        int height = (int)(screenSize.height*0.8);
        int width = (int)(height*1.25);
        
        String[] arguments = {"--center=true", 
                              "--width="+width, 
                              "--height="+height,
                              "--maximized=false"}; // ,"--fullScreen=true" 
        
        DesktopApplicationContext.main(CharacterCube.class, arguments);
    }
    
    // Desktop application & Applet
    @Override
    public void startup(Display display, Map<String, String> properties) throws Exception {
        
        //
        // Design
        //
        
        // If Applet
        screenSize = Toolkit.getDefaultToolkit().getScreenSize();
              
        if (screenSize.height < 1024) {
            border = 30;
            charSize = 12;
            titleSize = 16;
        }
        else if (screenSize.height < 1200) {
            border = 40;
            charSize = 14;
            titleSize = 18;
        }
        
        Theme theme = Theme.getTheme();
        
        Font themeFont = theme.getFont();
        textFont = themeFont.deriveFont((float)charSize);
        titleFont = themeFont.deriveFont((float)titleSize);
        
        //
    	// Java 3D universe
        //
        
        universe = new CharacterCubeUniverse(this);
        
        // 3D Canvas : PCanvas3D
        
        // Parent of PCanvas3D
        Panel overlayCanvas3DPanel = new Panel();      
        // PCanvas3D
        final PCanvas3D pCanvas3D = universe.createPCanvas3D(display, overlayCanvas3DPanel); 
                       
        // Overlay        
        fpsValueLable = new Label("FPS : 000");
        fpsValueLable.getStyles().put("font", titleFont);
        fpsValueLable.getStyles().put("color", Color.WHITE);
        fpsValueLable.getStyles().put("horizontalAlignment", HorizontalAlignment.LEFT);
        fpsValueLable.setSize(fpsValueLable.getPreferredSize()); 
        fpsValueLable.setLocation(25, 25);

        // Border
        final Border canvas3DBorder = new Border();
        canvas3DBorder.getStyles().put("color", bgPanel);
        canvas3DBorder.getStyles().put("padding", 0);
        canvas3DBorder.getStyles().put("thickness", border);
        
        // PCanvas3D fills the border component
        overlayCanvas3DPanel.getComponentListeners().add(new ComponentListener.Adapter() {
            @Override
            public void sizeChanged(Component component, int previousWidth,
                                                         int previousHeight) {
                pCanvas3D.setSize(component.getSize());                
            }
        });

        overlayCanvas3DPanel.add(fpsValueLable);
        
        canvas3DBorder.setContent(overlayCanvas3DPanel);

        //
        // Actions
        //
        
        // Title
        Label titleLabel = new Label("Character Cube");
        titleLabel.getStyles().put("font", titleFont);
        titleLabel.getStyles().put("color", Color.WHITE);
        titleLabel.getStyles().put("horizontalAlignment", HorizontalAlignment.CENTER);
        
        // Rotation
        
        BoxPane rotationBoxPane = new BoxPane();  
        rotationBoxPane.setOrientation(Orientation.VERTICAL);
        rotationBoxPane.getStyles().put("horizontalAlignment", HorizontalAlignment.CENTER);
        rotationBoxPane.getStyles().put("fill", true);

        // Reset rotation       
        PushButton rotationButton = new PushButton("Reset");
        
        // Direction, speed
        Label rotationLabel = new Label("<        0        >");
        rotationLabel.getStyles().put("horizontalAlignment", HorizontalAlignment.CENTER);
        
        final Slider rotationSlider = new Slider();
        rotationSlider.setRange(0, 100);
        rotationSlider.setValue(50);
        rotationSlider.getSliderValueListeners().add(new SliderValueListener() {
            public void valueChanged(Slider slider, int previousValue) {
                universe.setRotationSpeed(slider.getValue());
            }
        });
        
        rotationButton.setAction(new Action() {
            public void perform(Component source) {
                universe.resetRotation();
                rotationSlider.setValue(50);
            }
        });
        
        rotationBoxPane.add(rotationButton);
        rotationBoxPane.add(rotationLabel);
        rotationBoxPane.add(rotationSlider);
        
        CubeExpander rotationExpander = new CubeExpander();
        rotationExpander.setTitle("Cube Rotation");
        rotationExpander.setContent(rotationBoxPane);
        
        // Transparency
        
        BoxPane transBoxPane = new BoxPane();  
        transBoxPane.setOrientation(Orientation.VERTICAL);
        transBoxPane.getStyles().put("horizontalAlignment", HorizontalAlignment.CENTER);
        transBoxPane.getStyles().put("fill", true);

        Slider transSlider = new Slider();
        transSlider.setRange(0, 100);
        transSlider.setValue(30);
        transSlider.getSliderValueListeners().add(new SliderValueListener() {
            public void valueChanged(Slider slider, int previousValue) {
                universe.setCubeTransparency(slider.getValue());
            }
        });
        
        transBoxPane.add(transSlider);
        
        CubeExpander transExpander = new CubeExpander();       
        transExpander.setTitle("Cube Transparency ");
        transExpander.setContent(transBoxPane);
        
        // Viewpoints

        ButtonPressListener vpListener = new ButtonPressListener () {
            public void buttonPressed(Button button) {
                String value = ((CubeButton)button).getValue();
                universe.setVantagePoint(value);
            }
        };

        CubeButton arrowsButton = new CubeButton("Arrows", "Back");
        arrowsButton.getButtonPressListeners().add(vpListener);
        
        CubeButton dingbatsButton = new CubeButton("Dingbats", "Front");
        dingbatsButton.getButtonPressListeners().add(vpListener);
        
        CubeButton latinButton = new CubeButton("Latin", "Right");
        latinButton.getButtonPressListeners().add(vpListener);
              
        CubeButton mathButton = new CubeButton("Math", "Left");
        mathButton.getButtonPressListeners().add(vpListener);

        CubeButton homeButton = new CubeButton("Home");
        homeButton.getButtonPressListeners().add(vpListener);

        final CubeButton centerButton = new CubeButton("Center");
        centerButton.getButtonPressListeners().add(vpListener);
        
        BoxPane vpBoxPane  = new BoxPane(Orientation.VERTICAL);
        vpBoxPane.getStyles().put("verticalAlignment", VerticalAlignment.CENTER);
        vpBoxPane.getStyles().put("fill", true);

        vpBoxPane.add(arrowsButton);
        vpBoxPane.add(dingbatsButton);
        vpBoxPane.add(latinButton);
        vpBoxPane.add(mathButton);
        vpBoxPane.add(homeButton);
        vpBoxPane.add(centerButton);
        
        CubeExpander vpExpander = new CubeExpander();
        vpExpander.setTitle("Viewpoint");
        vpExpander.setContent(vpBoxPane);
        
        // Projection mode
        
        final ButtonGroup projGroup = new ButtonGroup();
        
        RadioButton perspectiveButton = new RadioButton("Perspective");
        perspectiveButton.getStyles().put("font", textFont);
        perspectiveButton.setSelected(true);
        //perspectiveButton.setGroup(projGroup);
        projGroup.add(perspectiveButton);

        RadioButton parallelButton = new RadioButton("Parallel");
        parallelButton.getStyles().put("font", textFont);
        //parallelButton.setGroup(projGroup);    
        projGroup.add(parallelButton);

        Action projAction = new Action () {
            public void perform(Component source) {               
                String value = (String)projGroup.getSelection().getButtonData();
                centerButton.setEnabled(value.equalsIgnoreCase("Perspective"));
                universe.setProjectionMode(value);
            }
        };
        perspectiveButton.setAction(projAction);
        parallelButton.setAction(projAction);
        
        BoxPane projBoxPane  = new BoxPane();
        projBoxPane.add(parallelButton);
        projBoxPane.add(perspectiveButton);
        projBoxPane.getStyles().put("spacing", 10);
        
        CubeExpander projExpander = new CubeExpander();
        projExpander.setTitle("Projection Mode");
        projExpander.setContent(projBoxPane);
        
        // Action panel
        
        BoxPane actionsBoxPane  = new BoxPane();  
        actionsBoxPane.setOrientation(Orientation.VERTICAL);
        actionsBoxPane.getStyles().put("fill", true);
        actionsBoxPane.getStyles().put("spacing", 10);

        actionsBoxPane.add(titleLabel);
        
        actionsBoxPane.add(rotationExpander);
        actionsBoxPane.add(transExpander);
        
        actionsBoxPane.add(projExpander);
        actionsBoxPane.add(vpExpander);
        
        BoxPane actionsSplitPane  = new BoxPane();  
        actionsSplitPane.setOrientation(Orientation.VERTICAL);
        actionsSplitPane.getStyles().put("backgroundColor", bgPanel);

        actionsSplitPane.add(actionsBoxPane);
        
        final Border actionSplitBorder = new Border();
        actionSplitBorder.getStyles().put("color", bgPanel);
        actionSplitBorder.getStyles().put("padding", 0);
        actionSplitBorder.getStyles().put("thickness", border);

        actionSplitBorder.setContent(actionsSplitPane);
        
        //
        // SplitPane
        // 
        
        SplitPane splitPane = new SplitPane();
        splitPane.getStyles().put("splitterThickness", 10);
        splitPane.getStyles().put("useShadow", true);
        splitPane.getStyles().put("backgroundColor", bgSplit);
        splitPane.setLeft(actionSplitBorder);
        splitPane.setRight(canvas3DBorder);
        
        float splitRatio = 0.3f;
        try {
            int leftWidth = actionsBoxPane.getPreferredWidth() + 2*border;
            int displayWidth = display.getDisplayHost().getParent().getWidth();
            splitRatio = (float)leftWidth / (float)displayWidth;
        }
        catch (NullPointerException e) {            
        }
        
        splitPane.setSplitRatio(splitRatio);
        
        //
        // Window
        //
        
        window = new Window();
        window.setTitle("InteractiveMesh : Java 3D meets Apache Pivot 2.0");
        window.setMaximized(true);
        window.setContent(splitPane);       
        window.open(display);
                     
        // FYI: system properties
        printSystemProps();
        universe.printJava3DProps();
    }

    @Override
    public boolean shutdown(boolean optional) {
        if (universe != null) {
            universe.closeUniverse();
        }
        if (window != null) {
            window.close();
        }

        return false;
    }

    @Override
    public void suspend() {
    }

    @Override
    public void resume() {
    }

    // Called from universe
    void updateFPSValue(int value) {
        fpsValueLable.setText("FPS : " + Integer.toString(value));
//        fpsValueLable.repaint();
    }
    
    //
    // GUI
    //
    
    private class CubeExpander extends Expander {
        
        private CubeExpander() {
          
            this.getStyles().put("backgroundColor", bgAction);
            
            TerraExpanderSkin skin = (TerraExpanderSkin)this.getSkin();
            skin.setTitleBarFont(textFont);
        }
    }
    
    private class CubeButton extends PushButton {
        
        private String value = null;
        
        private CubeButton (String text) {
            this(text, text);
        }
        
        private CubeButton (String text, String value) {
            super(text);
            this.value = value;
            this.getStyles().put("font", textFont);
        }
        
        private String getValue() {
            return value;
        }
        
    }
    
    // System properties
    private void printSystemProps() {
        System.out.println("");
        System.out.println("------------------------------------------------------------------------");
        System.out.println("SYSTEM  PROPERTIES");
        System.out.println("------------------------------------------------------------------------");
        System.out.println("OS Name  =  " + System.getProperty("os.name"));
        System.out.println("OS Arch  =  " + System.getProperty("os.arch"));
        System.out.println("OS Version  =  " + System.getProperty("os.version"));
        System.out.println("------------------------------------------------------------------------");
        System.out.println("Java Version  =  " + System.getProperty("java.version"));
        System.out.println("Java Vender  =  " + System.getProperty("java.vendor"));
        System.out.println("------------------------------------------------------------------------");
    }
}
