package com.interactivemesh.j3d.testspace.canvas3d;

/*
 * j3d-examples: src.resources.images.bg.jpg
 *
 * Copyright (c) 2007 Sun Microsystems, Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistribution of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * - Redistribution in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the
 *   distribution.
 *
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * This software is provided "AS IS," without a warranty of any
 * kind. ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND
 * WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY
 * EXCLUDED. SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL
 * NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF
 * USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
 * DERIVATIVES. IN NO EVENT WILL SUN OR ITS LICENSORS BE LIABLE FOR
 * ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE THIS SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * You acknowledge that this software is not designed, licensed or
 * intended for use in the design, construction, operation or
 * maintenance of any nuclear facility.
 *
 */

import java.awt.Color;
import java.awt.GraphicsConfiguration;
import java.awt.GraphicsEnvironment;

import java.awt.image.BufferedImage;

import java.io.FileNotFoundException;
import java.io.IOException;

import java.net.URL;

import java.util.Enumeration;
import java.util.Map;

import javax.imageio.ImageIO;

import javax.media.j3d.Alpha;
import javax.media.j3d.Background;
import javax.media.j3d.Behavior;
import javax.media.j3d.BoundingSphere;
import javax.media.j3d.Bounds;
import javax.media.j3d.BranchGroup;
import javax.media.j3d.Canvas3D;
import javax.media.j3d.DirectionalLight;
import javax.media.j3d.GraphicsConfigTemplate3D;
import javax.media.j3d.Group;
import javax.media.j3d.ImageComponent2D;
import javax.media.j3d.Locale;
import javax.media.j3d.PhysicalBody;
import javax.media.j3d.PhysicalEnvironment;
import javax.media.j3d.SceneGraphObject;
import javax.media.j3d.Shape3D;
import javax.media.j3d.Transform3D;
import javax.media.j3d.TransformGroup;
import javax.media.j3d.View;
import javax.media.j3d.ViewPlatform;
import javax.media.j3d.ViewSpecificGroup;
import javax.media.j3d.VirtualUniverse;
import javax.media.j3d.WakeupCriterion;
import javax.media.j3d.WakeupOnBehaviorPost;
import javax.media.j3d.WakeupOnElapsedFrames;
import javax.media.j3d.WakeupOr;

import javax.vecmath.Color3f;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;

// Loader Interface
import com.sun.j3d.loaders.IncorrectFormatException;
import com.sun.j3d.loaders.Loader;
import com.sun.j3d.loaders.ParsingErrorException;
import com.sun.j3d.loaders.Scene;
// VRML loader
//import org.jdesktop.j3d.loaders.vrml97.VrmlLoader;
//X3DModelImporter
import com.interactivemesh.j3d.interchange.ext3d.XImportException;
import com.interactivemesh.j3d.interchange.ext3d.XShapeReader;

// OrbitBehaviorInterim 2.1, see http://www.interactivemesh.org/testspace/orbitbehavior.html
import com.interactivemesh.j3d.community.utils.navigation.orbit.OrbitBehaviorInterim;

/*
 * PropellerUniverse.java
 *
 * Version: 2.0
 * Date: 2012/08/26
 *
 * Copyright (c) 2009-2012
 * August Lammersdorf, InteractiveMesh e.K.
 * Hauptstrae 28d, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * but you may not modify, compile, or distribute this 'PropellerUniverse.java'.
 *
 * See also license notice above: j3d-examples: src.resources.images.bg.jpg
 */

final class PropellerUniverse {

    static {
        System.out.println("\nPropellerUniverse : Copyright (c) 2009-2012 August Lammersdorf, www.InteractiveMesh.com.");
    }

    private BoundingSphere      globalBounds        =   null;

    private Locale              locale              =   null;

    private View                view                =   null;
    private View                viewRod             =   null;
    private View                viewPiston          =   null;
    private View                viewOnRod           =   null;

    private Canvas3D            canvas3D            =   null;
    private Canvas3D            canvas3DRod         =   null;
    private Canvas3D            canvas3DPiston      =   null;
    private Canvas3D            canvas3DOnRod       =   null;
    
    private OrbitBehaviorInterim orbitBehProp       =   null;
    private OrbitBehaviorInterim orbitBehRod        =   null;
    private OrbitBehaviorInterim orbitBehPiston     =   null;
    private OrbitBehaviorInterim orbitBehOnRod      =   null;

    private BranchGroup         sceneBranch         =   null;
    private BranchGroup         viewBranch          =   null;
    private BranchGroup         enviBranch          =   null;
    
    private TransformGroup      viewOnRodTG         =   null;

    private PropellerBehavior   propellerBehavior   =   null;
//    private Alpha               propellerAlpha      =   null;
//    private long                propellerStartTime  =   0;
    
    private FPSBehavior         fpsBehavior         =   null;
    private RepaintBehavior		repaintBehavior		=	null;

    private Color               bgColor0            =   new Color(0, 102, 204);
    
    private boolean				isD3D				=	false;
    private int					minCycleTime		=	0;

    private PropellerUniversePanel universePanel    =   null;
    
    enum Vp {
        FRONT("Front View"),
        PISTON("Piston View"),
        ROD("Rod View"),
        ONROD("Rod's View");
        
        Vp(String name) {
            vpName = name;
        }
        
        String vpName = null;
        
        private Canvas3D canvas3D = null;
        private OrbitBehaviorInterim orbitBeh = null;
        private VantagePoint vPoint = null;
    }
    
    PropellerUniverse(PropellerUniversePanel panel) {
    	
        // Not secure system property. 'set' and 'get' require signed jars 
        // when used on web in JWS or Applet. -Dj3d.rend=d3d  
        try {
            String prop = System.getProperty("j3d.rend");
            if (prop != null)
            	isD3D = prop.equals("d3d");
            if (isD3D)
            	minCycleTime = 8;		// 125 fps, due to Directx/JRE 7
        } 
        catch (SecurityException exception) {
        }

        
        universePanel = panel;
        
        createVantagePoints();

        createUniverse();
        createScene();
        setLive();

        // Setup navigation

        double sceneRadius = 1.0f;

        Bounds bounds = sceneBranch.getBounds();
        BoundingSphere sphereBounds = null;

        if (bounds.isEmpty()) {
            sphereBounds = new BoundingSphere();
        }
        else {
            if (bounds instanceof BoundingSphere)
                sphereBounds = (BoundingSphere)bounds;
            else
                sphereBounds = new BoundingSphere(bounds);
        }

        sceneRadius = sphereBounds.getRadius();

        orbitBehProp.setTransFactors(sceneRadius/4.0f, sceneRadius/4.0f);
        orbitBehProp.setZoomFactor(sceneRadius/4.0f);
        orbitBehProp.setRotFactors(0.4f, 0.4f);
        orbitBehProp.setClippingBounds(sphereBounds);

        orbitBehRod.setTransFactors(sceneRadius/4.0f, sceneRadius/4.0f);
        orbitBehRod.setZoomFactor(sceneRadius/4.0f);
        orbitBehRod.setRotFactors(0.4f, 0.4f);
        orbitBehRod.setClippingBounds(sphereBounds);

        orbitBehPiston.setTransFactors(sceneRadius/4.0f, sceneRadius/4.0f);
        orbitBehPiston.setZoomFactor(sceneRadius/4.0f);
        orbitBehPiston.setRotFactors(0.4f, 0.4f);
        orbitBehPiston.setClippingBounds(sphereBounds);

        orbitBehOnRod.setTransFactors(sceneRadius/4.0f, sceneRadius/4.0f);
        orbitBehOnRod.setZoomFactor(sceneRadius/4.0f);
        orbitBehOnRod.setRotFactors(0.4f, 0.4f);
        orbitBehOnRod.setClippingBounds(sphereBounds);

        setVantagePoint(Vp.FRONT);
        setVantagePoint(Vp.ONROD);
        setVantagePoint(Vp.PISTON);
        setVantagePoint(Vp.ROD);
        
printJava3DProps();
    }
    
    Canvas3D getCanvas3D(Vp vp) {
        return vp.canvas3D;
    }
    
    boolean isD3D() {
    	return isD3D;
    }
    
    //
    // Scene interaction
    //
    
    void setNavigatorEnable(boolean enable, Vp vp) {
         vp.orbitBeh.setEnable(enable);
    }

    void setMinFrameCycleTime(long minimumTime) {
        view.setMinimumFrameCycleTime(minimumTime);
        viewRod.setMinimumFrameCycleTime(minimumTime);
        viewPiston.setMinimumFrameCycleTime(minimumTime);
        viewOnRod.setMinimumFrameCycleTime(minimumTime);
    }

    // 
    void setVantagePoint(Vp vp) {
        vp.vPoint.applyTo(vp.orbitBeh);           
    }
    
    void startStopRepainter(boolean start) {
    	if (start)
    		repaintBehavior.postId(RepaintBehavior.REPAINT_START);
    	else
    		repaintBehavior.postId(RepaintBehavior.REPAINT_STOP);
    }

    void resetRotation() {
    	propellerBehavior.postId(50);
    }

    void setRotationSpeed(int speed) {
    	propellerBehavior.postId(speed);
    }

    void closeUniverse() {
        
        view.removeAllCanvas3Ds();
        view.attachViewPlatform(null);
        locale.getVirtualUniverse().removeAllLocales();
        
        /**/
        viewRod.removeAllCanvas3Ds();
        viewRod.attachViewPlatform(null);
        viewPiston.removeAllCanvas3Ds();
        viewPiston.attachViewPlatform(null);        
        viewOnRod.removeAllCanvas3Ds();
        viewOnRod.attachViewPlatform(null);

    }

    //
    // Create scene
    //
    private void createScene() {

        // X3D importer
        XShapeReader x3dImporter = new XShapeReader();
        // Read 
        x3dImporter.read(this.getClass().getResource("resources/PropellerModelOCC-AllShapes.x3d"));
      
        // Imported named SceneGraphObjects
        Map<String, SceneGraphObject> namedShapes = x3dImporter.getNamedSGOs();
      
        x3dImporter.close();

        Shape3D crankarm1    = (Shape3D)namedShapes.get("Crankarm");
        Shape3D crankarm0    = (Shape3D)namedShapes.get("PistonWristPin");
        Shape3D engineBlock  = (Shape3D)namedShapes.get("EngineBlock");
        Shape3D cylinderhead = (Shape3D)namedShapes.get("CylinderHead");
        Shape3D piston       = (Shape3D)namedShapes.get("Piston");
        Shape3D propAxis     = (Shape3D)namedShapes.get("PropAxis");
        Shape3D propeller    = (Shape3D)namedShapes.get("Propeller");

        TransformGroup propellerGroup = new TransformGroup();
        propellerGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
        TransformGroup propPositionGroup = new TransformGroup();
        Transform3D t3d = new Transform3D();
        t3d.rotZ(Math.toRadians(45));
        propPositionGroup.setTransform(t3d);
        propPositionGroup.addChild(propeller);
        propellerGroup.addChild(propPositionGroup);
        propellerGroup.addChild(propAxis);
       
        TransformGroup crankarmGroup = new TransformGroup();
        crankarmGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
        crankarmGroup.addChild(crankarm1);
        crankarmGroup.addChild(viewOnRodTG);
        
        TransformGroup cylinderGroup = new TransformGroup();
        cylinderGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
        cylinderGroup.addChild(crankarmGroup);
        cylinderGroup.addChild(crankarm0);
        cylinderGroup.addChild(piston);
        
        Group engineblockGroup = new Group();
        engineblockGroup.addChild(engineBlock);
        engineblockGroup.addChild(cylinderhead);
   
        // Propeller assembly
        Group modelGroup = new Group();
        modelGroup.addChild(propellerGroup);
        modelGroup.addChild(cylinderGroup);
        modelGroup.addChild(engineblockGroup);

        propellerBehavior = new PropellerBehavior();
        propellerBehavior.setSchedulingBounds(globalBounds);        
        propellerBehavior.crankarmTG = crankarmGroup;
        propellerBehavior.cylinderTG = cylinderGroup;
        propellerBehavior.propellerTG = propellerGroup;
        
        fpsBehavior = new FPSBehavior();
        fpsBehavior.setSchedulingBounds(globalBounds);
        
        repaintBehavior = new RepaintBehavior();
        repaintBehavior.setSchedulingBounds(globalBounds);

        // Add to scene
        sceneBranch.addChild(propellerBehavior);
        sceneBranch.addChild(fpsBehavior);
        sceneBranch.addChild(repaintBehavior);
        sceneBranch.addChild(modelGroup);
    }
    
    // Set live
    private void setLive() {
        locale.addBranchGraph(sceneBranch);
        locale.addBranchGraph(viewBranch);
        locale.addBranchGraph(enviBranch);
    }

    // Base world
    private void createUniverse() {
        
        // Bounds
        globalBounds = new BoundingSphere();
        globalBounds.setRadius(Double.MAX_VALUE);
        
        //
        // Viewing
        //
        
        PhysicalBody phBody = new PhysicalBody();
        PhysicalEnvironment phEnvironment = new PhysicalEnvironment();
        
        view = new View();
        view.setPhysicalBody(phBody);
        view.setPhysicalEnvironment(phEnvironment);
view.setMinimumFrameCycleTime(minCycleTime); 
        
        viewRod = new View();
        viewRod.setPhysicalBody(phBody);
        viewRod.setPhysicalEnvironment(phEnvironment);
viewRod.setMinimumFrameCycleTime(minCycleTime); 
        
        viewPiston = new View();
        viewPiston.setPhysicalBody(phBody);
        viewPiston.setPhysicalEnvironment(phEnvironment);
viewPiston.setMinimumFrameCycleTime(minCycleTime);
        
        viewOnRod = new View();
        viewOnRod.setPhysicalBody(phBody);
        viewOnRod.setPhysicalEnvironment(phEnvironment);
viewOnRod.setMinimumFrameCycleTime(minCycleTime);
        
        GraphicsConfigTemplate3D gCT = new GraphicsConfigTemplate3D();
        GraphicsConfiguration gcfg = GraphicsEnvironment.getLocalGraphicsEnvironment().
                                        getDefaultScreenDevice().getBestConfiguration(gCT);
        try {
            canvas3D = new Canvas3D(gcfg);
            canvas3DRod = new Canvas3D(gcfg);
            canvas3DPiston = new Canvas3D(gcfg);
            canvas3DOnRod = new Canvas3D(gcfg);
            
            Vp.FRONT.canvas3D = canvas3D;
            Vp.PISTON.canvas3D = canvas3DRod;
            Vp.ROD.canvas3D = canvas3DPiston;
            Vp.ONROD.canvas3D = canvas3DOnRod;
        }
        catch (Exception e) {
            System.out.println("PropellerUniverse: canvas3D failed !!");
            e.printStackTrace();
            System.exit(0);
        }

        //
        // SuperStructure
        //
        VirtualUniverse vu = new VirtualUniverse();
        locale = new Locale(vu);
        //
        // BranchGraphs
        //
        sceneBranch = new BranchGroup();
        viewBranch = new BranchGroup();
        enviBranch = new BranchGroup();

        // ViewBranch

        TransformGroup viewTG = new TransformGroup();
        viewTG.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
        ViewPlatform vp = new ViewPlatform();
        view.attachViewPlatform(vp);
        view.addCanvas3D(canvas3D);

        TransformGroup viewRodTG = new TransformGroup();
        viewRodTG.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
        ViewPlatform vpRod = new ViewPlatform();
        viewRod.attachViewPlatform(vpRod);
        viewRod.addCanvas3D(canvas3DRod);

        TransformGroup viewPistonTG = new TransformGroup();
        viewPistonTG.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
        ViewPlatform vpPiston = new ViewPlatform();
        viewPiston.attachViewPlatform(vpPiston);
        viewPiston.addCanvas3D(canvas3DPiston);

        viewOnRodTG = new TransformGroup();
        viewOnRodTG.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
        ViewPlatform vpOnRod = new ViewPlatform();
        viewOnRod.attachViewPlatform(vpOnRod);
        viewOnRod.addCanvas3D(canvas3DOnRod);
        
        // 
        orbitBehProp = new OrbitBehaviorInterim(canvas3D, viewTG, view, OrbitBehaviorInterim.REVERSE_ALL);
        orbitBehProp.setSchedulingBounds(globalBounds);
        orbitBehProp.setClippingEnabled(true);        

        orbitBehRod = new OrbitBehaviorInterim(canvas3DRod, viewRodTG, viewRod, OrbitBehaviorInterim.REVERSE_ALL);
        orbitBehRod.setSchedulingBounds(globalBounds);
        orbitBehRod.setClippingEnabled(true);

        orbitBehPiston = new OrbitBehaviorInterim(canvas3DPiston, viewPistonTG, viewPiston, OrbitBehaviorInterim.REVERSE_ALL);
        orbitBehPiston.setSchedulingBounds(globalBounds);
        orbitBehPiston.setClippingEnabled(true);

        orbitBehOnRod = new OrbitBehaviorInterim(canvas3DOnRod, viewOnRodTG, viewOnRod, OrbitBehaviorInterim.REVERSE_ALL);
        orbitBehOnRod.setSchedulingBounds(globalBounds);
        orbitBehOnRod.setClippingEnabled(true);
        
        Vp.FRONT.orbitBeh = orbitBehProp;
        Vp.PISTON.orbitBeh = orbitBehRod; // !!
        Vp.ROD.orbitBeh = orbitBehPiston; // !!
        Vp.ONROD.orbitBeh = orbitBehOnRod;

        ViewSpecificGroup propVSG = new ViewSpecificGroup();
        ViewSpecificGroup rodVSG = new ViewSpecificGroup();
        ViewSpecificGroup pistonVSG = new ViewSpecificGroup();
        ViewSpecificGroup onRodVSG = new ViewSpecificGroup();
        
        DirectionalLight headLightProp = new DirectionalLight();
        headLightProp.setInfluencingBounds(globalBounds);

        DirectionalLight headLightRod = new DirectionalLight();
        headLightRod.setInfluencingBounds(globalBounds);

        DirectionalLight headLightPiston = new DirectionalLight();
        headLightPiston.setInfluencingBounds(globalBounds);

        DirectionalLight headLightOnRod = new DirectionalLight();
        headLightOnRod.setInfluencingBounds(globalBounds);
        
        
        propVSG.addChild(headLightProp);
        propVSG.addView(view);
        
        rodVSG.addChild(headLightRod);
        rodVSG.addView(viewRod);
        
        pistonVSG.addChild(headLightPiston);
        pistonVSG.addView(viewPiston);

        onRodVSG.addChild(headLightOnRod);
        onRodVSG.addView(viewOnRod);
        

        viewTG.addChild(vp);
        viewTG.addChild(orbitBehProp);
        viewTG.addChild(propVSG);

        viewRodTG.addChild(rodVSG);
        viewRodTG.addChild(vpRod);
        viewRodTG.addChild(orbitBehRod);
        
        viewPistonTG.addChild(pistonVSG);
        viewPistonTG.addChild(vpPiston);
        viewPistonTG.addChild(orbitBehPiston);

        viewOnRodTG.addChild(onRodVSG);
        viewOnRodTG.addChild(vpOnRod);
        viewOnRodTG.addChild(orbitBehOnRod);
        
        
        viewBranch.addChild(viewTG);
        viewBranch.addChild(viewRodTG);
        viewBranch.addChild(viewPistonTG);

        // EnviBranch

        final Background background = new Background();
        background.setCapability(Background.ALLOW_IMAGE_WRITE);
        background.setApplicationBounds(globalBounds);
        background.setColor(new Color3f(bgColor0));

        URL imageUrl = this.getClass().getResource("resources/bg.jpg");
        try {
            BufferedImage bgImage = ImageIO.read(imageUrl);
            if (bgImage != null) {
                background.setImageScaleMode(Background.SCALE_FIT_ALL);
                background.setImage(new ImageComponent2D(ImageComponent2D.FORMAT_RGB, bgImage, false, false));
            }
        }
        catch (IOException e) {
        }
        
        enviBranch.addChild(background);
    }
    
    private void createVantagePoints() {
        
        Vector3d upVector  = new Vector3d(0.0, 1.0, 0.0);
        Point3d  rotCenter = new Point3d(0, 0, -0.0525);
        double   fov       = Math.toRadians(45); // default value is used
        //                                    name   -    VP's position         -        look at           -           up vector - center of rotation   -      fov angle
        Vp.FRONT.vPoint  = new VantagePoint("Front",     new Point3d(0.72, 0, 1.73),      new Point3d(-0.25, 0, -0.0525),  upVector, rotCenter,                    fov);
        Vp.ONROD.vPoint  = new VantagePoint("OnRod",     new Point3d(0.09, 0, -0.46),       new Point3d(0.09, 0, 0),      upVector, rotCenter,                    fov);
        Vp.ROD.vPoint    = new VantagePoint("Pistonrod", new Point3d(-0.159, 0, -0.268),     new Point3d(0.045, 0, -0.1275), upVector, new Point3d(0.045, 0, -0.1275), fov);
        Vp.PISTON.vPoint = new VantagePoint("Piston",    new Point3d(0.1087, 0.023, -0.1038), new Point3d(0, 0, -0.10378),   upVector, new Point3d(0, 0, -0.10378),   fov);
    }

    private final class VantagePoint {

        private String      name        =   null;

        private Point3d     position =  new Point3d();
        private Point3d     viewCenter  =   new Point3d();
        private Vector3d    up          =   new Vector3d();
        private Point3d     rotCenter   =   new Point3d();
        //private double      fov           =   Math.PI/4;

        private VantagePoint(String name,
                             Point3d position,
                             Point3d viewCenter,
                             Vector3d up,
                             Point3d rotationCenter, double fov) {
            this.name = name;

            this.position.set(position);
            this.viewCenter.set(viewCenter);
            this.up.set(up);
            this.rotCenter.set(rotationCenter);
            //this.fov = fov;

        }

        @Override
        public String toString() {
            return name;
        }

        private void applyTo(OrbitBehaviorInterim navigator) {
            
            navigator.setViewingTransform(position, viewCenter, up, rotCenter);
            
            //navigator.setFieldOfView(fov);
        }
    }
    
    private Shape3D loadShape3D(String filename, Loader vrmlLoader) {
        
        URL sceneUrl = this.getClass().getResource("resources/" + filename);
        if (sceneUrl == null) {
            return null;
        }

        Shape3D shape3D = null;
        
        try {
            Scene scene = vrmlLoader.load(sceneUrl);
            BranchGroup sceneGroup = scene.getSceneGroup();
            shape3D = (Shape3D)sceneGroup.getChild(0);            
            sceneGroup.removeAllChildren();           
        }
        catch (FileNotFoundException e) {
            e.printStackTrace();
        }
        catch (IncorrectFormatException e) {
            e.printStackTrace();
        }
        catch (ParsingErrorException e) {
            e.printStackTrace();
        }

        return shape3D;
    }
    
    private final class PropellerBehavior extends Behavior {
        
        private double          onePi                   =   Math.PI;
        private double          twoPi                   =   Math.PI * 2;

        private double          axisRadians             =   0.0;
        private double 			sinAxisRadians 			= 	0.0;
        private double          crankarmRadians         =   0.0;

        private double          radius                  =   0.045;
        private double          crankarmLength          =   0.120;
        private double          dist                    =   0.165; // 0.045 + 0.120
        private double          currDist                =   0.165;
        private double          transX                  =   0.0;
        
        // Tarnsformation
        private TransformGroup  cylinderTG              =   null;
        private Transform3D     cylinderTransform       =   new Transform3D();
        private Vector3d        cylinderTranslation     =   new Vector3d();

        private TransformGroup  propellerTG             =   null;
        private Transform3D     propellerRotation       =   new Transform3D();

        private TransformGroup  crankarmTG              =   null;
        private Transform3D     crankarmTransform       =   new Transform3D();
        private Transform3D     crankarmRotation        =   new Transform3D();
        private Transform3D     crankarmCenterTrans     =   new Transform3D();
        private Transform3D     crankarmCenterTransInv  =   new Transform3D();
        private Transform3D     crankarmTemp            =   new Transform3D();
        
        // "Alpha"
        private long 			NsecPerMsec 			= 	1000000L; // nano sec per millisecond
        private long 			MinDurationCW			= 	200L; // 0.2 sec per revolution
        private long 			MinDurationCCW			= 	100L; // 0.1 sec per revolution
        private long 			startTime 				= 	0L; // for a specific speed
        private long 			duration 				= 	0L; // of a single revolution
        private float 			currAlphaValue 			= 	0f; // [0, 1]
        private float 			loops 					= 	0f; // revolutions since startTime
        
        private boolean 		isCCWrotation			=	false;
                
        // WakeupCriterions
        private WakeupOnBehaviorPost 	speedPost 		= 	null;    
        private WakeupOr 				perFramesOrPost	= 	null;
        
        private WakeupCriterion	currCriterion			=	null;

        private PropellerBehavior() {

            crankarmCenterTrans.setTranslation(new Vector3d(-0.165, 0, 0));
            crankarmCenterTransInv.setTranslation(new Vector3d(0.165, 0, 0));
            
            speedPost = new WakeupOnBehaviorPost(this, 0);
            WakeupCriterion[] wcs = new WakeupCriterion[]{new WakeupOnBehaviorPost(this, 0), new WakeupOnElapsedFrames(0)};
            perFramesOrPost = new WakeupOr(wcs);
        }
        
        @Override
        public void initialize() {
            wakeupOn(speedPost);
        }

        @Override
        public void processStimulus(Enumeration criteria) {
        	
        	while (criteria.hasMoreElements()) { 
        		
        		currCriterion = (WakeupCriterion)criteria.nextElement();
        		
        		if (currCriterion instanceof WakeupOnElapsedFrames) {
        			
                    // [0, 1] : fraction of current loop
                    //currAlphaValue = ((System.nanoTime - startTime) % duration) / duration  
                    loops = (System.nanoTime() - startTime); 
                    loops /= duration;
                    currAlphaValue = loops - (int)loops;
                    if (isCCWrotation)
                    	currAlphaValue = 1 - currAlphaValue;

                    axisRadians = twoPi * currAlphaValue;

                    sinAxisRadians = Math.sin(axisRadians);

                    // Law of sines

                    crankarmRadians = Math.asin(radius * sinAxisRadians / crankarmLength);

                    // If currRadians != onePi | twoPi | 0
                    if (sinAxisRadians != 0.0f) {
                        currDist = crankarmLength / sinAxisRadians * Math.sin(onePi-axisRadians-crankarmRadians);
                    }
                    else {
                        if (axisRadians == onePi) {
                            currDist = crankarmLength - radius;
                        }
                        else {
                            currDist = crankarmLength + radius;
                        }
                    }

                    transX = dist - currDist;
                    
                    cylinderTranslation.setX(-transX);
                    cylinderTransform.setTranslation(cylinderTranslation);
                    cylinderTG.setTransform(cylinderTransform);

                    // Rotates in (0, 0, 0)
                    propellerRotation.rotZ(axisRadians);
                    propellerTG.setTransform(propellerRotation);

                    // Rotates in (0.165, 0, 0)
                    crankarmRotation.rotZ(-crankarmRadians);
                    crankarmTemp.mul(crankarmRotation, crankarmCenterTrans);
                    crankarmTransform.mul(crankarmCenterTransInv, crankarmTemp);
                    crankarmTG.setTransform(crankarmTransform);
                    
                    wakeupOn(perFramesOrPost);
                }
        		else if (currCriterion instanceof WakeupOnBehaviorPost) {
        			
        			int speed = ((WakeupOnBehaviorPost)currCriterion).getTriggeringPostId(); // [0, 100]
        			
                    // Stop rotation
                    if (speed > 48 && speed < 52) { 
                    	wakeupOn(speedPost);   
                    	return;
                    }
                    // Set speed: slider [1, 100] 
                    else {
                    	if (speed > 51) {
                    		speed -= 51;
                    		isCCWrotation = false;
                        	// duration -> speed
                        	duration = (long)(NsecPerMsec * MinDurationCW * 50f/speed);  	  
                        	// startTime -> sync
                        	startTime = System.nanoTime() - (long)(currAlphaValue * duration);
                    	}
                    	else {
                    		speed = 49 - speed;
                    		isCCWrotation = true;
                        	// duration -> speed
                        	duration = (long)(NsecPerMsec * MinDurationCCW * 50f/speed);  	  
                        	// startTime -> sync
                        	startTime = System.nanoTime() - (long)((1 - currAlphaValue) * duration);
                    	}
                      
                    	wakeupOn(perFramesOrPost);
                    }       			
        		}
        		else 
        			wakeupOn(speedPost); // should not happen !!
        	}
        }
    }
    
    private final class FPSBehavior extends Behavior {
        
        private WakeupOnElapsedFrames   onFramesP       =   null;
        
        private double                  frameDuration   =   0;      
        private long                    startTime       =   0;      
        private int                     frameCounter    =   0;      
        private int                     elapsedFrames   =   10;
        
        private FPSBehavior() {
            onFramesP = new WakeupOnElapsedFrames(0, true);
        }
        
        @Override
        public void initialize() {
            wakeupOn(onFramesP);
        }
        @Override
        public void processStimulus(Enumeration criteria) {         
                    
            frameCounter++;
            
            if (frameCounter >= elapsedFrames) {
                
                long stopTime = System.nanoTime();
                frameDuration = ((double)(stopTime - startTime))/frameCounter;
                
                float fps = (float)(1000000000d / frameDuration);
                
                startTime = stopTime;

                frameCounter = 0;
                elapsedFrames = (int)Math.max(1, ((fps + 0.5f)/4)); // 4 times per second
                
                if (universePanel != null)
                    universePanel.setFPS((int)fps);
            }
            
            wakeupOn(onFramesP);                                        
        }       
    }
    
	private final class RepaintBehavior extends Behavior {
		
		private static final int REPAINT_START = 1;
		private static final int REPAINT_STOP  = 2;
		private WakeupOnBehaviorPost 	startPost 		= 	null;				
        private WakeupOr 				perFramesOrStop	= 	null;

		private WakeupCriterion	currCriterion			=	null;
		
		private RepaintBehavior() {			
            WakeupCriterion[] wcs = new WakeupCriterion[]{new WakeupOnBehaviorPost(this, REPAINT_STOP), 
            		                                      new WakeupOnElapsedFrames(0)};
            perFramesOrStop = new WakeupOr(wcs);
            startPost = new WakeupOnBehaviorPost(this, REPAINT_START);
		}
		
		@Override
		public void initialize() {
			wakeupOn(startPost);
		}
		@Override
		public void processStimulus(Enumeration criteria) {					
        	while (criteria.hasMoreElements()) { 
        		
        		currCriterion = (WakeupCriterion)criteria.nextElement();
        		
        		if (currCriterion instanceof WakeupOnElapsedFrames) {
	    			wakeupOn(perFramesOrStop);
	            }
	    		else if (currCriterion instanceof WakeupOnBehaviorPost) {
        			if (((WakeupOnBehaviorPost)currCriterion).getTriggeringPostId() == REPAINT_START) {
        				wakeupOn(perFramesOrStop);
        			}
        			else if (((WakeupOnBehaviorPost)currCriterion).getTriggeringPostId() == REPAINT_STOP) {        				
        				wakeupOn(startPost);
        				return;
        			}        		
	    		}
    		}
		}
	}
	
	// Java 3D properties
	private void printJava3DProps() {
		System.out.println("");
		System.out.println("------------------------------------------------------------------------");
		System.out.println("SYSTEM  PROPERTIES");
		System.out.println("------------------------------------------------------------------------");
		System.out.println("OS Name  =  " + System.getProperty("os.name"));
		System.out.println("OS Arch  =  " + System.getProperty("os.arch"));
		System.out.println("OS Version  =  " + System.getProperty("os.version"));
		System.out.println("------------------------------------------------------------------------");
		System.out.println("Java Version  =  " + System.getProperty("java.version"));
		System.out.println("Java Vender  =  " + System.getProperty("java.vendor"));
		System.out.println("------------------------------------------------------------------------");

		Map vuProps = VirtualUniverse.getProperties();

	    System.out.println("Java 3D Version  =  " + vuProps.get("j3d.version"));
	    System.out.println("Java 3D Renderer  =  " + vuProps.get("j3d.renderer"));
	    System.out.println("Java 3D Pipeline  =  " + vuProps.get("j3d.pipeline"));
	    System.out.println("------------------------------------------------------------------------");
	  
	    Map c3dProps = canvas3D.queryProperties();

		System.out.println("Native Version  =  " + c3dProps.get("native.version"));
	    System.out.println("Native GLSL  =  " + c3dProps.get("shadingLanguageGLSL"));
	    System.out.println("Native Vendor  =  " + c3dProps.get("native.vendor"));
	    System.out.println("Native Renderer  =  " + c3dProps.get("native.renderer"));

	    System.out.println("------------------------------------------------------------------------");
	    System.out.println("");
	}

}