package com.interactivemesh.j3d.testspace.awtshape;

import java.awt.Font;
import java.awt.GraphicsConfiguration;
import java.awt.GraphicsEnvironment;
import java.awt.Shape;

import java.awt.geom.GeneralPath;

import javax.media.j3d.Appearance;
import javax.media.j3d.Background;
import javax.media.j3d.BoundingSphere;
import javax.media.j3d.Bounds;
import javax.media.j3d.BranchGroup;
import javax.media.j3d.Canvas3D;
import javax.media.j3d.DirectionalLight;
import javax.media.j3d.Font3D;
import javax.media.j3d.FontExtrusion;
import javax.media.j3d.GeometryArray;
import javax.media.j3d.GraphicsConfigTemplate3D;
import javax.media.j3d.Locale;
import javax.media.j3d.Material;
import javax.media.j3d.PhysicalBody;
import javax.media.j3d.PhysicalEnvironment;
import javax.media.j3d.Shape3D;
import javax.media.j3d.Text3D;
import javax.media.j3d.Transform3D;
import javax.media.j3d.TransformGroup;
import javax.media.j3d.View;
import javax.media.j3d.ViewPlatform;
import javax.media.j3d.VirtualUniverse;

import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;

import com.interactivemesh.j3d.community.utils.geometry.AWTShapeExtruder;
import com.interactivemesh.j3d.community.utils.geometry.AWTShapeExtrusion;
import com.interactivemesh.j3d.community.utils.geometry.String3D;

import com.interactivemesh.j3d.community.utils.navigation.orbit.OrbitBehaviorInterim;

/*
 * S3DT3DComparison
 * 
 * Author: August Lammersdorf, www.InteractiveMesh.com/org
 * Version: 1.1
 * Date: 2008/12/11 
 * 
 * Please create your own implementation.
 * You are allowed to copy all lines you like of this source code, 
 * but you may not modify, compile, or distribute this 'S3DT3DComparison'. 
 *
*/

final class S3DT3DComparison {
     
    static {
        System.out.println("S3DT3DComparison: Copyright (c) 2008 August Lammersdorf, www.InteractiveMesh.com.");
    }

    private BoundingSphere          globalBounds    =   null;

    private Canvas3D                canvas3D        =   null;
    private Locale                  locale          =   null;

    private BranchGroup             sceneBranch     =   null;
    private BranchGroup             viewBranch      =   null;
    private BranchGroup             enviBranch      =   null;

    private OrbitBehaviorInterim    orbitBehInterim =   null;
    
    S3DT3DComparison() {

        createUniverse();
        createScene();
        setLive();

        // Setup navigation

        float sceneRadius = 1.0f;

        Bounds bounds = sceneBranch.getBounds();
        BoundingSphere sphereBounds = null; 

        if (bounds.isEmpty()) {
            sphereBounds = new BoundingSphere(); 
        }
        else {
            if (sphereBounds instanceof BoundingSphere)
                sphereBounds = (BoundingSphere)sceneBranch.getBounds();
            else
                sphereBounds = new BoundingSphere(sceneBranch.getBounds());

            sceneRadius = (float)sphereBounds.getRadius();
        }
        
        orbitBehInterim.setProjectionMode(View.PARALLEL_PROJECTION);
        
        orbitBehInterim.setTransFactors(sceneRadius/2.0f, sceneRadius/2.0f);
        orbitBehInterim.setZoomFactor(sceneRadius/2.0f);
        orbitBehInterim.setRotFactors(0.75f, 0.75f);

        orbitBehInterim.setClippingBounds(sphereBounds);

        orbitBehInterim.goHome(true);
    }

    Canvas3D getCanvas3D() {
        return canvas3D;
    }

    // Extrusion shape
    private Shape createExtrShape(float extr, float extend) {
        
        float dx = 0.05f;
        float dy = 0.1f * extend;
        GeneralPath gp = new GeneralPath();
        gp.moveTo(0, 0);
        gp.lineTo(dx, 1.15f*dy);
        gp.curveTo((extr+2*dx)*0.4f, 2.0f*dy,  (extr+2*dx)*0.6f, 2.0f*dy,  extr+dx, 1.15f*dy);
        gp.lineTo(extr+2*dx, 0);

        return gp;
    }
     
    // Text3D geometry
    private Text3D createText3D(String text, Shape extrShape, Font font) {
        
        FontExtrusion fontExtr = new FontExtrusion(extrShape, 0.002);              
        Font3D        font3D   = new Font3D(font, 0.001, fontExtr);
        
        Text3D text3D = new Text3D(font3D, text);
        text3D.setCharacterSpacing(0.3f);

        return text3D;
    }
    
    // String3D geometry
    private GeometryArray createString3D(String text, Shape extrShape, Font font, boolean t3dStyle) {
       
       AWTShapeExtrusion extrusion = new AWTShapeExtrusion(extrShape, 0.002); 
       extrusion.setT3DStyleExtrusion(t3dStyle);
       AWTShapeExtruder  extruder  = new AWTShapeExtruder(0.001, extrusion, Math.toRadians(20));
       
       String3D string3D = new String3D(font, extruder);
       string3D.setCharacterSpacing(0.3f);

       return string3D.getStringGeometry(text);
    }
   
    private void createScene() {
        
        Appearance appearT3D = new Appearance();
        Material matT3D = new Material();
        matT3D.setAmbientColor(0.0f, 0.0f, 0.0f);
        matT3D.setDiffuseColor(0.0f, 0.35f, 0.7f);
        matT3D.setEmissiveColor(0.0f, 0.15f, 0.3f);
        matT3D.setSpecularColor(0.0f, 0.2f, 0.6f);
        matT3D.setShininess(4.0f);        
        appearT3D.setMaterial(matT3D);
        
        Appearance appearS3D = new Appearance();
        Material matS3D = new Material();
        matS3D.setAmbientColor(0.0f, 0.0f, 0.0f);
        matS3D.setDiffuseColor(0.0f, 0.6f, 0.6f);
        matS3D.setEmissiveColor(0.3f, 0.2f, 0.2f);
        matS3D.setSpecularColor(0.0f, 0.2f, 0.2f);
        matS3D.setShininess(4.0f);        
        appearS3D.setMaterial(matS3D);
        
        Appearance appear = new Appearance();
        Material mat = new Material();
        mat.setAmbientColor(0.0f, 0.0f, 0.0f);
        mat.setDiffuseColor(0.7f, 0.0f, 0.0f);
        mat.setEmissiveColor(0.3f, 0.0f, 0.0f);
        mat.setSpecularColor(0.0f, 0.7f, 0.04f);
        mat.setShininess(8.0f);        
        appear.setMaterial(mat);

        // Font, size 1 m
        Font font = new Font("SansSerif", Font.PLAIN, 1); // V 1.1
        
        // Critical characters
        String text = "Xpjht";
        
        // Extrusion shape                  depth, side extent factor
        Shape extrShape_0 = createExtrShape(0.75f, 0.8f);
        Shape extrShape_1 = createExtrShape(0.75f, 0.4f);
        Shape extrShape_2 = createExtrShape(0.75f, 0.2f);
        
        // Text3D
        
        Shape3D textT3DShape3D = new Shape3D();    
        textT3DShape3D.setAppearance(appearT3D);
        textT3DShape3D.setGeometry(createText3D("T3D: ", extrShape_2, font));
        
        Shape3D textShape3D_0 = new Shape3D();    
        textShape3D_0.setAppearance(appear);
        textShape3D_0.setGeometry(createText3D(text, extrShape_0, font));

        Shape3D textShape3D_1 = new Shape3D();    
        textShape3D_1.setAppearance(appear);
        textShape3D_1.setGeometry(createText3D(text, extrShape_1, font));

        Shape3D textShape3D_2 = new Shape3D();    
        textShape3D_2.setAppearance(appear);
        textShape3D_2.setGeometry(createText3D(text, extrShape_2, font));
        
        Transform3D t3d = new Transform3D();
        
        TransformGroup tg_0 = new TransformGroup();       
        t3d.setTranslation(new Vector3d(4, 0, 0));
        tg_0.setTransform(t3d);
                
        TransformGroup tg_1 = new TransformGroup();
        t3d.setTranslation(new Vector3d(8, 0, 0));
        tg_1.setTransform(t3d);
        
        TransformGroup tg_2 = new TransformGroup();
        t3d.setTranslation(new Vector3d(12, 0, 0));
        tg_2.setTransform(t3d);
        
        tg_0.addChild(textShape3D_0);
        tg_1.addChild(textShape3D_1);
        tg_2.addChild(textShape3D_2);
        
        sceneBranch.addChild(textT3DShape3D);
        sceneBranch.addChild(tg_0);
        sceneBranch.addChild(tg_1);
        sceneBranch.addChild(tg_2);
        
        // String3D - Text3D style
        
        Shape3D stringS3DT3DShape3D = new Shape3D();    
        stringS3DT3DShape3D.setAppearance(appearS3D);
        stringS3DT3DShape3D.setGeometry(createString3D("S3D: ", extrShape_2, font, true));
        
        Shape3D stringT3DShape3D_0 = new Shape3D();    
        stringT3DShape3D_0.setAppearance(appear);
        stringT3DShape3D_0.setGeometry(createString3D(text, extrShape_0, font, true));

        Shape3D stringT3DShape3D_1 = new Shape3D();    
        stringT3DShape3D_1.setAppearance(appear);
        stringT3DShape3D_1.setGeometry(createString3D(text, extrShape_1, font, true));

        Shape3D stringT3DShape3D_2 = new Shape3D();    
        stringT3DShape3D_2.setAppearance(appear);
        stringT3DShape3D_2.setGeometry(createString3D(text, extrShape_2, font, true));

        TransformGroup tg_S3DT3D = new TransformGroup();       
        t3d.setTranslation(new Vector3d(0, -1.5f, 0));
        tg_S3DT3D.setTransform(t3d);

        TransformGroup tg_3 = new TransformGroup();       
        t3d.setTranslation(new Vector3d(4, -1.5f, 0));
        tg_3.setTransform(t3d);
                
        TransformGroup tg_4 = new TransformGroup();
        t3d.setTranslation(new Vector3d(8, -1.5f, 0));
        tg_4.setTransform(t3d);
        
        TransformGroup tg_5 = new TransformGroup();
        t3d.setTranslation(new Vector3d(12, -1.5f, 0));
        tg_5.setTransform(t3d);
        
        tg_S3DT3D.addChild(stringS3DT3DShape3D);
        tg_3.addChild(stringT3DShape3D_0);
        tg_4.addChild(stringT3DShape3D_1);
        tg_5.addChild(stringT3DShape3D_2);
        
        sceneBranch.addChild(tg_S3DT3D);
        sceneBranch.addChild(tg_3);
        sceneBranch.addChild(tg_4);
        sceneBranch.addChild(tg_5);

        // String3D 
        
        Shape3D stringS3DShape3D = new Shape3D();    
        stringS3DShape3D.setAppearance(appearS3D);
        stringS3DShape3D.setGeometry(createString3D("S3D: ", extrShape_2, font, false));
        
        Shape3D stringShape3D_0 = new Shape3D();    
        stringShape3D_0.setAppearance(appear);
        stringShape3D_0.setGeometry(createString3D(text, extrShape_0, font, false));

        Shape3D stringShape3D_1 = new Shape3D();    
        stringShape3D_1.setAppearance(appear);
        stringShape3D_1.setGeometry(createString3D(text, extrShape_1, font, false));

        Shape3D stringShape3D_2 = new Shape3D();    
        stringShape3D_2.setAppearance(appear);
        stringShape3D_2.setGeometry(createString3D(text, extrShape_2, font, false));

        TransformGroup tg_S3D = new TransformGroup();       
        t3d.setTranslation(new Vector3d(0, -3, 0));
        tg_S3D.setTransform(t3d);

        TransformGroup tg_6 = new TransformGroup();       
        t3d.setTranslation(new Vector3d(4, -3, 0));
        tg_6.setTransform(t3d);
                
        TransformGroup tg_7 = new TransformGroup();
        t3d.setTranslation(new Vector3d(8, -3, 0));
        tg_7.setTransform(t3d);
        
        TransformGroup tg_8 = new TransformGroup();
        t3d.setTranslation(new Vector3d(12, -3, 0));
        tg_8.setTransform(t3d);
        
        tg_S3D.addChild(stringS3DShape3D);
        tg_6.addChild(stringShape3D_0);
        tg_7.addChild(stringShape3D_1);
        tg_8.addChild(stringShape3D_2);
        
        sceneBranch.addChild(tg_S3D);
        sceneBranch.addChild(tg_6);
        sceneBranch.addChild(tg_7);
        sceneBranch.addChild(tg_8);
    }
    
    private void setLive() {
        locale.addBranchGraph(sceneBranch);
        locale.addBranchGraph(viewBranch);
        locale.addBranchGraph(enviBranch);
    }
    
    private void createUniverse() {     
        
        // Bounds
        globalBounds = new BoundingSphere();
        globalBounds.setRadius(Double.MAX_VALUE);  
        
        //
        // Viewing
        //
        View view = new View();
        view.setPhysicalBody(new PhysicalBody());
        view.setPhysicalEnvironment(new PhysicalEnvironment());
        
        GraphicsConfigTemplate3D gCT = new GraphicsConfigTemplate3D();
        GraphicsConfiguration gcfg = GraphicsEnvironment.getLocalGraphicsEnvironment().getDefaultScreenDevice().getBestConfiguration(gCT);
        
        try {
            canvas3D = new Canvas3D(gcfg);
        }
        catch (NullPointerException e) {
            System.out.println("S3DT3DComparison: Canvas3D failed !!");
            e.printStackTrace();
            System.exit(0);
        }
        catch (IllegalArgumentException e) {
            System.out.println("S3DT3DComparison: Canvas3D failed !!");
            e.printStackTrace();
            System.exit(0);
        }               
        
        view.addCanvas3D(canvas3D);

        //
        // SuperStructure
        //
        VirtualUniverse vu = new VirtualUniverse();
        locale = new Locale(vu);        
        
        //
        // BranchGraphs
        //      
        sceneBranch = new BranchGroup();
        viewBranch = new BranchGroup();
        enviBranch = new BranchGroup();
        
        // ViewBranch
        
        TransformGroup viewTG = new TransformGroup();
        viewTG.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
                       
        ViewPlatform vp = new ViewPlatform();
        view.attachViewPlatform(vp);
        
        DirectionalLight headLight = new DirectionalLight();
        headLight.setInfluencingBounds(globalBounds);

        viewTG.addChild(vp);
        viewTG.addChild(headLight);
        
        viewBranch.addChild(viewTG);
        
        // Navigator
        orbitBehInterim = new OrbitBehaviorInterim(canvas3D, viewTG, view, OrbitBehaviorInterim.REVERSE_ALL);
        orbitBehInterim.setSchedulingBounds(globalBounds);
        orbitBehInterim.setClippingEnabled(true);
        Transform3D homeTransform = new Transform3D();
        homeTransform.setTranslation(new Vector3d(7.87f, -1.2f, 20));
        orbitBehInterim.setHomeTransform(homeTransform);        
        orbitBehInterim.setHomeRotationCenter(new Point3d(7.87f, -1.2f, 0));

        viewTG.addChild(orbitBehInterim);       
        
        // EnviBranch
        
        Background bg = new Background();
        bg.setApplicationBounds(globalBounds);
        bg.setColor(0.2f, 0.4f, 1.0f);
        
        enviBranch.addChild(bg);
    }
}
