package org.interactivemesh.scala.j3d.samples.distortstring

// Java
import java.awt.{AlphaComposite, Color, GradientPaint, Graphics, Graphics2D, GraphicsDevice}
import java.util.concurrent.TimeUnit
// Java 3D
import javax.media.j3d.GraphicsConfigTemplate3D
// ScalaCanvas3D API 1.0, see http://www.interactivemesh.org/testspace/j3dmeetsscala.html
import org.interactivemesh.scala.swing.j3d.SJCanvas3DAbstract

/*
 * DistortStringSJCanvas3D.scala
 *
 * Version: 1.2
 * Date: 2011/05/26
 *
 * Copyright (c) 2010-2011
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * but you may not modify, compile, or distribute this 'DistortStringSJCanvas3D.scala'.
 *
 */

/**
 * DistortString specific lightweight Java 3D rendering target.
 * Composite 2D and 3D rendering.
 *
 */
private[distortstring] class DistortStringSJCanvas3D(
    device: GraphicsDevice,
    template: GraphicsConfigTemplate3D,
    private val panel: DistortStringPanel)
      extends SJCanvas3DAbstract(device, template) {

  require(panel ne null, "DistortStringPanel is null !")
  
  //
  // Reflection
  //
    
  private var alphaComposite: AlphaComposite = AlphaComposite.SrcOver // default: SrcOver
  private var gradPaint: GradientPaint = null
  private var reflectFraction: Float = 0.75f
  private var tOpacity: Float = 0.1f
  private var bOpacity: Float = 0.8f
  private val topCol: Array[Float] = Array(0, 0, 0)
  private val botCol: Array[Float] = Array(0, 0, 0)
  private var reflectHeight: Int = 0

  //
  // Frames per second
  //
    
  private val updatesPerSecond: Int = 4
  private var elapsedFrames: Int = 50
  private var frameCounter: Int = 0
  private var startTimePaint: Long = System.nanoTime()/1000000
  private var endTimePaint: Long = 0L
  private var frameDuration: Long = 0L
  
  //
  // Reflection API
  //
    
  def fraction: Float = {
    reflectFraction
  }
  def fraction_=(f: Float): Unit = {
    reflectFraction = f
    // Reflection setup: requires new off-screen buffers
    createOffScreenBuffer(this.size.width, this.size.height)
  }
  
  def fractionHeight: Int = reflectHeight
    
  def topOpacity: Float = {
    tOpacity
  }
  def topOpacity_=(o: Float) {
    tOpacity = o
    createGradient   
    repaint
  }

  def bottomOpacity: Float = {
    bOpacity
  }
  def bottomOpacity_=(o: Float) {
    bOpacity = o
    createGradient  
    repaint
  }

  def topColorRGB: Color = {
    new Color(topCol(0), topCol(1), topCol(2))
  }
  def topColorRGB_=(c: Color) {
    c.getColorComponents(topCol)
    createGradient    
    repaint
  }

  def bottomColorRGB: Color = {
    new Color(botCol(0), botCol(1), botCol(2))
  }
  def bottomColorRGB_=(c: Color) {
    c.getColorComponents(botCol)
    createGradient   
    repaint
  }

  private def createGradient {
    gradPaint = new GradientPaint(
      0, 0,             new Color(topCol(0), topCol(1), topCol(2), tOpacity),              
      0, reflectHeight, new Color(botCol(0), botCol(1), botCol(2), bOpacity))
  }    
  
  //
  // SJCanvas3DAbstract
  //
  
  override protected def createOffScreenBuffer(panelWidth: Int, panelHeight: Int) {   
	  
    // 1. Canvas.height = panel.height - reflection.height >= panel.height/2
    
    reflectHeight = (panelHeight * 0.5f * reflectFraction + 0.5f).asInstanceOf[Int]
    
    val canvasHeight = panelHeight - reflectHeight
           
    // 2. GradientPaint
    createGradient

    // At last !!
    super.createOffScreenBuffer(panelWidth, canvasHeight)
  }

  /**
   * Callback used to allow an overriding subclass to execute individual code
   * when new off-screen buffers were created. 
   * <p>
   * This method is called internally by the event-dispatching thread (EDT)
   * and should not be called by applications. 
   * </p>
   */
  override protected def offScreenBufferCreated() {
	  
  }

  /**
   * Callback used to allow an overriding subclass to execute individual code
   * when the off-screen buffer was copied.
   * <p>
   * This method is called internally by the event-dispatching thread (EDT)
   * and should not be called by applications. 
   * </p>
   */
  override protected def offScreenBufferCopied() {
	  
  }

  // 2D and 3D painting, FPS
  override protected def paintComponent(g2d: Graphics2D) {
    peer match {
      case peer: CanvasSuperMixin => super.paintComponent(g2d)
            
        // Ready for drawing ?
        if (!isReadyForDrawing) {
            return
        }
      
        var isLocked: Boolean = false

        try {
          // Avoids deadlock when J3D-Renderer thread acquires the lock 
          // in 'postRender' but don't call 'postSwap' due to J3D internal states
          isLocked = imageAccessLock.tryLock(50, TimeUnit.MILLISECONDS)
    
          // Draw & flip scene image at the top
          g2d.drawImage(paintImage,
                    // destination in g2d: flip lowerY and upperY 
                    // dx1    dy1          dx2         dy2
                        0, imageHeight, imageWidth, 0,    
                    // source: the bottom part of the scene image
                    // sx1    sy1          sx2         sy2
                        0, 0,           imageWidth, imageHeight, null)
            
          // Reflection
          if (reflectHeight > 0) {
            // Draw mirrored scene image below, image is already yUp 
            g2d.drawImage(paintImage,
                      // destination in g2d
                      // dx1    dy1          dx2         dy2
                          0, imageHeight, imageWidth, this.size.height,    
                      // source: the upper part of the scene image
                      // sx1    sy1          sx2        sy2
                          0, 0,           imageWidth, reflectHeight, null)
                                
            // Move to mirrored scene image
            g2d.translate(0, imageHeight)
                
            // Set gradient
            g2d.setPaint(gradPaint)
            // Set composite
            g2d.setComposite(alphaComposite)
                
            // Draw gradient according current AlphaComposite state
            // on top of the mirrored scene image
            g2d.fillRect(0, 0, imageWidth, reflectHeight)
          }
            
          // Release J3D-Renderer thread
          isImageDrawn = true
          if (isLocked)
          	imagePaintCondition.signal()
        }         
        finally  {               
          if (isLocked)
            imageAccessLock.unlock()
        }
        
        // Frames per second

        frameCounter += 1

        if (frameCounter >= elapsedFrames) {

          endTimePaint = System.nanoTime()/1000000
          frameDuration = (endTimePaint-startTimePaint)/frameCounter           
	      startTimePaint = endTimePaint
	
	      var fpsPaint: Int = 0
          if (frameDuration > 0) {
	        fpsPaint = (1000 / frameDuration).asInstanceOf[Int]
          }           
          panel.updateFPS(fpsPaint)

          // Reset
          frameCounter = 0
          elapsedFrames = Math.max(1, ((fpsPaint + 0.5) / updatesPerSecond)).asInstanceOf[Int]
        }
      
      case _ => {}
    }
  }

}

