package org.interactivemesh.scala.j3d.samples.distortstring

// Java
import java.awt.{Color, Font, Graphics2D}
import java.awt.font.{FontRenderContext, GlyphVector, LineMetrics}

import java.util.{Enumeration => JEnumeration}

// Java 3D
import javax.media.j3d.{Alpha, Appearance, Background, Behavior, BoundingBox,
  BoundingSphere, Bounds, BranchGroup, Canvas3D, DirectionalLight, Geometry,
  GeometryArray, GeometryUpdater, GLSLShaderProgram, ImageComponent, ImageComponent2D,
  Locale, Material, PhysicalBody, PhysicalEnvironment, Shader, ShaderAppearance,
  ShaderAttributeObject, ShaderAttributeSet, ShaderAttributeValue, ShaderProgram,
  Shape3D, SourceCodeShader, Switch, TexCoordGeneration, Texture, Texture2D,
  TextureAttributes, Transform3D, TransformGroup, TriangleArray, View, ViewPlatform, 
  VirtualUniverse, WakeupCriterion, WakeupOnBehaviorPost, WakeupOnElapsedFrames, WakeupOr}

import javax.vecmath.{AxisAngle4d, Color3f, Point3d, Point3f, Vector3d, Vector3f, Vector4f}

import com.sun.j3d.utils.shader.StringIO

// AWTShapeExtruder API 3.0, see: http://www.interactivemesh.org/testspace/awtshapeextruder.html
import com.interactivemesh.j3d.community.utils.geometry.{AWTShapeExtruder, AWTShapeExtrusion, String3D}

// OrbitBehaviorInterim API 2.1, see http://www.interactivemesh.org/testspace/orbitbehavior.html
import com.interactivemesh.j3d.community.utils.navigation.orbit.OrbitBehaviorInterim

// ScalaCanvas3D API 1.0, see http://www.interactivemesh.org/testspace/j3dmeetsscala.html
import org.interactivemesh.scala.swing.j3d.SJCanvas3DAbstract

/*
 * DistortStringUniverse.scala
 *
 * Version: 1.2
 * Date: 2011/05/26
 *
 * Copyright (c) 2010-2011
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * but you may not modify, compile, or distribute this 'DistortStringUniverse.scala'.
 *
 * Special license for 'distortBehavior' ! 
 */

private final class DistortStringUniverse(private val panel: DistortStringPanel) {
	
  require(panel ne null, "No panel instance !!")

  private val bgColor = new Color(0.0f, 0.4f, 0.8f)
  private val globalBounds = new BoundingSphere(new Point3d, java.lang.Double.MAX_VALUE)
	
  private var isGLSL   = false
  private var isWoodShader = false
  
  //
  // SuperStructure
  //
  
  private val vu = new VirtualUniverse
  private val locale = new Locale(vu)

  //
  // Viewing
  //
  
  private val view = new View {
    setPhysicalBody(new PhysicalBody)
    setPhysicalEnvironment(new PhysicalEnvironment)
    setBackClipPolicy(View.VIRTUAL_EYE)
    setFrontClipPolicy(View.VIRTUAL_EYE)
    setBackClipDistance(15)
    setFrontClipDistance(0.005)
  }
  
  private val orbitBehavior = new OrbitBehaviorInterim(OrbitBehaviorInterim.REVERSE_ALL) {
    setSchedulingBounds(globalBounds)
    setVpView(DistortStringUniverse.this.view) // due to 'view'-naming collision in OrbitBehaviorInterim
  }
  
  //
  // BranchGraphs
  //
  
  private val enviBranch = new BranchGroup
  private val sceneBranch = new BranchGroup
  private val viewBranch = new BranchGroup
  
  // View branch
  
  viewBranch.addChild(
    new TransformGroup {
	  setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE) 
	
	  private val vP = new ViewPlatform
	
      view.attachViewPlatform(vP)
    
      addChild(vP)
      addChild(orbitBehavior)
      addChild( new DirectionalLight { setInfluencingBounds(globalBounds) } )
    
      orbitBehavior.setViewingTransformGroup(this)
    }
  )
  
  // Environment branch  
  
  private val bg = new Background {
    setCapability(Background.ALLOW_IMAGE_WRITE)
    setApplicationBounds(globalBounds)
    setColor(new Color3f(bgColor))
  }
  
  enviBranch.addChild(bg)
  enviBranch.addChild(bgImageBehavior)
  enviBranch.addChild(panelBehavior)

  //
  // Scene
  //
  
  // String3D Geometry
 
  import String3D._
  
  private val string3D = new String3D {
    
    // GeneralPath is final
    private val extend = 3.0f
    private val depth = 50.0f
    private val cut = depth/10.0f  
    private val path = new java.awt.geom.GeneralPath // GeneralPath is final
    path.moveTo(0.0f, 0.0f)
    path.lineTo(cut, extend)
    path.lineTo(depth-cut, extend)
    path.lineTo(depth, 0.0f)
	  
    private val scaleT3D = new Transform3D {
      setScale(0.01)                             // scale down font size to 1 with 1/100
      setTranslation(new Vector3d(0, 0, -0.25))  // center z
    }
    
    private val extruder = new AWTShapeExtruder {
      setCreaseAngle(Math.toRadians(24))
	  setGeometryTransform(scaleT3D)
	  setTessellationTolerance(0.075)
	  setShapeExtrusion(new AWTShapeExtrusion(path))
    }  
	  
    setFont(new Font("Dialog", Font.PLAIN, 100))
    setExtruder(extruder)
    setPosition(new Point3f(0, 0, 0))
    setCharacterSpacing(2.0f)
    setAlignment(Alignment.CENTER)
    setPath(Path.RIGHT)
  }

  private val string3DBox = new BoundingBox

  // Wood geometry
  
  private val woodGeom = string3D.getStringGeometry("Scala", string3DBox)
  private val vertexCt = woodGeom.getVertexCount
  
//println("vertexCt = " + vertexCt)

  // Gold geometry
  
  import GeometryArray._
  
  private object goldGeom extends TriangleArray(
      vertexCt,
      COORDINATES | NORMALS | TEXTURE_COORDINATE_2 | BY_REFERENCE) {
	  
    // Duplicate coordinates and normals from 'woodGeom'
    private val coords = new Array[Float](vertexCt*3)
    private val normals = new Array[Float](vertexCt*3)

    woodGeom.getCoordinates(0, coords)
    woodGeom.getNormals(0, normals)
	  
    private val lower = new Point3d
    private val upper = new Point3d
    string3DBox.getLower(lower)
    string3DBox.getUpper(upper)

    // Texture coordinates
    private val xScale = 1.0/(upper.x-lower.x)
    private val yScale = 1.0f/(upper.y-lower.y)
    private val planeSx = xScale.asInstanceOf[Float]
    private val planeSw = (-lower.x * xScale).asInstanceOf[Float]
    private val planeTy = yScale.asInstanceOf[Float]
    private val planeTw = (-lower.y * yScale).asInstanceOf[Float]

    private[DistortStringUniverse] val planeS = new Vector4f(planeSx, 0f, 0f, planeSw)
    private[DistortStringUniverse] val planeT = new Vector4f(0.0f, planeTy, 0f, planeTw)

    private val texCoords = new Array[Float](vertexCt*2)

    private var t = 0
    for (i <- 0 until vertexCt) {
      texCoords(t) = coords(i*3)*planeSx + planeSw
      t += 1
      texCoords(t) = coords(i*3 + 1)*planeTy + planeTw
      t += 1
    }
	  	  	  
    setCapability(GeometryArray.ALLOW_REF_DATA_WRITE)
    setCoordRefFloat(coords)
    setNormalRefFloat(normals)
    setTexCoordRefFloat(0, texCoords)
  }
  
  // Texture modes
  private[distortstring] object TexMode extends Enumeration {
    type TexMode = Value
    val Gold1, Gold2, Gold3, Wood1, Wood2 = Value
  }
  
  import TexMode._

  // Gold appearance 
  
  private object goldAppearance extends Appearance {
        
    setCapability(Appearance.ALLOW_TEXGEN_WRITE)

    // Default white diffuse color for modulate texture blending
    setMaterial(new Material {
      setSpecularColor(0.0f, 0.0f, 0.0f)
    })

    // Gold texture
    private val imageUrl = this.getClass.getResource("gold.jpg")
    try {
      val texImage = javax.imageio.ImageIO.read(imageUrl)
      if (texImage ne null) {
        setTexture(new Texture2D( 
          Texture.BASE_LEVEL, Texture.RGB, texImage.getWidth, texImage.getHeight) {
            setMagFilter(Texture.NICEST)
            setMinFilter(Texture.NICEST)
        	setImage(0, new ImageComponent2D(ImageComponent.FORMAT_RGB, texImage))
          }
        )      
      }
    }
    catch {
      case e: java.io.IOException => {}
    }

    // TexCoordGenerations: OBJECT_LINEAR | SPHERE_MAP texture coordinates
    import TexCoordGeneration._
    
    private val texCoordGenLinear = new TexCoordGeneration(OBJECT_LINEAR, TEXTURE_COORDINATE_2) {
        setPlaneS(goldGeom.planeS)
        setPlaneT(goldGeom.planeT)
      }

    private val texCoordGenSphere =  new TexCoordGeneration(SPHERE_MAP, TEXTURE_COORDINATE_2)
                
    setTexCoordGeneration(texCoordGenSphere)    // at start time

    // TextureAttributes: MODULATE | REPLACE blending
    import TextureAttributes._
    
    private val texAttr = new TextureAttributes {
      setCapability(ALLOW_MODE_WRITE)
      setTextureMode(REPLACE)                   // at start time
    }
        
    setTextureAttributes(texAttr)
    
    // 
    private[DistortStringUniverse]def varyTexture(mode: TexMode): Boolean = {
      var done = true
	  mode match {
	    case Gold1 =>
          setTexCoordGeneration(texCoordGenSphere)
          texAttr.setTextureMode(REPLACE)    
	    case Gold2 =>
          setTexCoordGeneration(texCoordGenLinear)
          texAttr.setTextureMode(MODULATE)   
	    case Gold3 =>
          setTexCoordGeneration(null)
          texAttr.setTextureMode(MODULATE)      
	     case _ => done = false
	  }
      return done
    }
  }
  
  // Wood appearance    
  
  private object woodAppearance extends ShaderAppearance {

    private var vertexProgram: String = ""
    private var fragmentProgram: String = ""
    try {
      vertexProgram = StringIO.readFully(this.getClass.getResource("WoodDistort.vert"))
      fragmentProgram = StringIO.readFully(this.getClass.getResource("WoodDistort.frag"))
    }
    catch {
    	case e: java.io.IOException => throw new RuntimeException(e)
    }
    
    import Shader._

    private val shaders = Array[Shader](
      new SourceCodeShader(SHADING_LANGUAGE_GLSL, SHADER_TYPE_VERTEX, vertexProgram),
      new SourceCodeShader(SHADING_LANGUAGE_GLSL, SHADER_TYPE_FRAGMENT, fragmentProgram)
    )
    
    private val shaderAttrNames = Array(
      "Spx",
      "Spy",
      "IsDistortTexture",
      "Scale", 
      "GrainSizeRecip",
      "DarkColor",
      "Spread"
    )
    
    private val shaderAttrValues = Array(
      0.0f,
      0.0f,
      0,   // Boolean  0? false : true
      2.0f, 
      5.0f, 
      new Color3f(0.6f, 0.3f, 0.1f),
      new Color3f(0.15f, 0.075f, 0.0f)
    )
    
    private val shaderProgram = new GLSLShaderProgram {
      setShaders(shaders)
      setShaderAttrNames(shaderAttrNames)
    }
    
    // Wood ShaderAttributeValues
    private val attrValueSpx = new ShaderAttributeValue(shaderAttrNames(0), shaderAttrValues(0)) {
      setCapability(ShaderAttributeObject.ALLOW_VALUE_WRITE)
    }
    private val attrValueSpy = new ShaderAttributeValue(shaderAttrNames(1), shaderAttrValues(1)) {
      setCapability(ShaderAttributeObject.ALLOW_VALUE_WRITE)
    }
    private val attrValueDistTex = new ShaderAttributeValue(shaderAttrNames(2), shaderAttrValues(2)) {
      setCapability(ShaderAttributeObject.ALLOW_VALUE_WRITE)
    }

    private val shaderAttributeSet = new ShaderAttributeSet {
      put(attrValueSpx)
      put(attrValueSpy)
      put(attrValueDistTex)
    }

    for (i <- 3 until shaderAttrNames.length) {
      shaderAttributeSet.put( new ShaderAttributeValue(shaderAttrNames(i),  shaderAttrValues(i)) )
    }

    setShaderAttributeSet(shaderAttributeSet)
    setShaderProgram(shaderProgram)
    
    //
    private[DistortStringUniverse] def attrValueSpXY(spx: Float, spy: Float) {
      attrValueSpx.setValue(spx)
      attrValueSpy.setValue(spy)
    }    
    // 
    private[DistortStringUniverse] def varyTexture(mode: TexMode): Boolean = {
      var done = true
      mode match {
	    case Wood1 =>
          attrValueDistTex.setValue(0)  // false
        case Wood2 =>
          attrValueDistTex.setValue(1)  // true
        case _ => done = false
      }
      return done
    }
  }  
  
  
  // Toggle shapes
  private val GOLD_SHAPE3D: Int = 0
  private val WOOD_SHAPE3D: Int = 1

  private val shapeSwitch = new Switch(GOLD_SHAPE3D) { // Switch.CHILD_NONE, GOLD_SHAPE3D WOOD_SHAPE3D
    setCapability(Switch.ALLOW_SWITCH_WRITE)
    addChild(new Shape3D(goldGeom, goldAppearance)) // Gold shape
    addChild(new Shape3D(woodGeom, woodAppearance)) // Wood shape
  }  

// TODO Conflict / bug : background image and shader !!??
// At first add workaround shape        
// workaround shape : a triangle with three identical vertices and TransparencyAttributes
  sceneBranch.addChild(
    new Shape3D(
      new TriangleArray(3, GeometryArray.COORDINATES) {
        setCoordinate(0, Array(0f, 0f, 0f, 0f, 0f, 0f, 0f, 0f, 0f))
      },
      new Appearance {   
        import javax.media.j3d.TransparencyAttributes
        setTransparencyAttributes(
          new TransparencyAttributes {
            setTransparencyMode(TransparencyAttributes.NICEST)
          }
        )
      }
    )
  )
    
  // Rotate 3D string due to distortBehavior's calculations
  sceneBranch.addChild(
    new TransformGroup {
      setTransform(new Transform3D {
        setEuler(new Vector3d(0, -Math.toRadians(80), 0))
      })     
      addChild(shapeSwitch)
    }
  )
    
  // distortBehavior
  sceneBranch.addChild(distortBehavior)
  
  // Set live
  
  locale.addBranchGraph(sceneBranch)
  locale.addBranchGraph(viewBranch)
  locale.addBranchGraph(enviBranch)
  
  //
  // Universe interaction
  //
  
  //
  private[distortstring] def addCanvasPanel(sjCanvas3D: SJCanvas3DAbstract): Unit = {
	
	val onScreenPanel = sjCanvas3D.onscreenJPanel
	val offScreenCanvas3D = sjCanvas3D.offscreenCanvas3D
	
    val c3dProps = offScreenCanvas3D.queryProperties
    isGLSL = (c3dProps.get("shadingLanguageGLSL")).asInstanceOf[Boolean]

	// Add offscreen Canvas3D to View
    view.addCanvas3D(offScreenCanvas3D)       
    
    // Avoids flickr
    onScreenPanel.setBackground(bgColor)
    
    // Setup navigation
    
    orbitBehavior.setAWTComponent(onScreenPanel)

    var sceneRadius = 1.0

    val bounds = sceneBranch.getBounds
    var sphereBounds = new BoundingSphere
    if (!bounds.isEmpty) {
      if (sphereBounds.isInstanceOf[BoundingSphere])
        sphereBounds = bounds.asInstanceOf[BoundingSphere]
      else
        sphereBounds = new BoundingSphere(bounds)
    }
        
    sceneRadius = sphereBounds.getRadius

    orbitBehavior.setTransFactors(sceneRadius/4.0, sceneRadius/4.0)
    orbitBehavior.setZoomFactor(sceneRadius/4.0)
    orbitBehavior.setRotFactors(0.5, 0.5)
    // Requires an added Canvas3D
    orbitBehavior.setProjectionMode(View.PARALLEL_PROJECTION)
    orbitBehavior.setPureParallelEnabled(true)
    
    // Initial viewpoint
    vantagePoint("Front")
    
    // Start distortion and string animation        
    startStopRebound(true)
    startStopDistort(true)
  }
  
  // Shader supported ?
  private[distortstring] def isGLSLavailable: Boolean = isGLSL
       
  private[distortstring] def setupBgImage(width: Int, height: Int): Unit =     
    bgImageBehavior.imageSize(width, height)
    
  private[distortstring] def vantagePoint(vp: String): Unit = {
    val vpTransform = new Transform3D
    if (vp.equalsIgnoreCase("Front")) {
      vpTransform.setTranslation(new Vector3d(0.0, 0.45, 4.0))
    }
    else if (vp.equalsIgnoreCase("Side")) {
      vpTransform.setRotation(new AxisAngle4d(0, 1, 0, Math.toRadians(-90)))
      vpTransform.setTranslation(new Vector3d(-4.0, 0.45, 0.0))
    }
    else
      return
    orbitBehavior.setViewingTransform(vpTransform, new Point3d)
  }

  private[distortstring] def startStopDistort(start: Boolean): Unit = distortBehavior.startStop(start) 
  private[distortstring] def speedDistort(s: Int): Unit = distortBehavior.speed(s)
  
  private[distortstring] def startStopRebound(start: Boolean): Unit = bgImageBehavior.startStop(start)
  private[distortstring] def speedRebound(s: Int): Unit = bgImageBehavior.speed(s)

  private[distortstring] def hideShowPanel(hide: Boolean): Unit = panelBehavior.hideShow(hide)

  // Vary textures   
  private[distortstring] def varyGoldTexture(mode: TexMode): Unit = {
    if (goldAppearance.varyTexture(mode) &&
    	shapeSwitch.getWhichChild != GOLD_SHAPE3D) {
      isWoodShader = false
      shapeSwitch.setWhichChild(GOLD_SHAPE3D)
    }
  }
  private[distortstring] def varyWoodTexture(mode: TexMode): Unit = {
    if (woodAppearance.varyTexture(mode) &&
    	isGLSL && shapeSwitch.getWhichChild != WOOD_SHAPE3D) {
      isWoodShader = true
      shapeSwitch.setWhichChild(WOOD_SHAPE3D)
    }
  }

  private[distortstring] def closeUniverse: Unit = {
    view.removeAllCanvas3Ds
    view.attachViewPlatform(null)
    vu.removeAllLocales
  }

  //
  // Behaviors
  //

  // Control panel : show/hide   
  private object panelBehavior extends Behavior {
    	
    setSchedulingBounds(globalBounds)
    	    	
    private val HIDE: Int = 0
    private val SHOW: Int = 1

    private val hideCriterion 	= new WakeupOnBehaviorPost(this, HIDE)
    private val showCriterion  	= new WakeupOnBehaviorPost(this, SHOW)
    private val frameCriterion  = new WakeupOnElapsedFrames(0)
    	
    private var isPanelShown = true      
        
    private val alpha = new Alpha {
      setIncreasingAlphaDuration(1000)       
      setDecreasingAlphaDuration(1000) 
      setLoopCount(1)       	
    }        
        
    private[DistortStringUniverse] def hideShow(hide: Boolean) {
      if (hide) postId(HIDE)
      else postId(SHOW)
    }
        
    // Behavior
    
    override def initialize = wakeupOn(hideCriterion)

    override def processStimulus(criteria: JEnumeration[_]) { // renamed java.util.Enumeration
      while (criteria.hasMoreElements) {    	  
        criteria.nextElement match {
        	
          // Panel animation
          case w: WakeupOnElapsedFrames =>
          
            panel.hideShowControls(alpha.value)
                	
            if (alpha.finished) {
              if (isPanelShown) 
            	wakeupOn(hideCriterion)
              else 
            	wakeupOn(showCriterion)
            }
            else 
              wakeupOn(frameCriterion)
            
          // Init hide/show
          case w: WakeupOnBehaviorPost =>
          
            w.getTriggeringPostId match {
              case HIDE =>
                isPanelShown = false
                alpha.setMode(Alpha.INCREASING_ENABLE)
                alpha.setStartTime(System.currentTimeMillis)
              case SHOW =>
                isPanelShown = true
                alpha.setMode(Alpha.DECREASING_ENABLE)
                alpha.setStartTime(System.currentTimeMillis)
              case _ =>                
            }
            wakeupOn(frameCriterion)
            
          // Shouldn't happen
          case _ => wakeupOn(frameCriterion)
        }
      }
    }  
  }
  
  // Background image, 2D string animation
  private object bgImageBehavior extends Behavior with ImageComponent2D.Updater {   	
  
    setSchedulingBounds(globalBounds)
	  
    private val MinDurationX = 4000L                                     // max speed
    private val MinDurationY = 2600L                                     // max speed
    private var newDurationX = (MinDurationX*100f/30).asInstanceOf[Long] // start speed
    private var newDurationY = (MinDurationY*100f/30).asInstanceOf[Long] // start speed
    	
    private val REZISEIMAGE  = 1
    private val START        = 98
    private val STOP         = 99
    
    private val resizeStart = new WakeupOr(
      Array(new WakeupOnBehaviorPost(this, START), new WakeupOnBehaviorPost(this, REZISEIMAGE))
    )    
    private val frameResizeStop = new WakeupOr(
      Array(new WakeupOnElapsedFrames(0), new WakeupOnBehaviorPost(this, REZISEIMAGE), new WakeupOnBehaviorPost(this, STOP))
    )
        
    private val startTime = System.currentTimeMillis
    
    private val xAlpha = new Alpha {
      setIncreasingAlphaDuration(newDurationX)       
      setLoopCount(-1) 
      setStartTime(startTime)
      pause
    }
    private val yAlpha = new Alpha {
      setIncreasingAlphaDuration(newDurationY)     
      setLoopCount(-1)
      setStartTime(startTime)
      pause
    }

    // Background image
    
    import java.awt.image.BufferedImage
    
    private var bgImage: BufferedImage = null
    private var bgImageComp: ImageComponent2D = null
    private var currImgWidth  = 1
    private var currImgHeight = 1
    private var newImgWidth   = 1
    private var newImgHeight  = 1
    
    // Animated string
    
    private	val chars: Array[Char] = new String("Scala Days 2011").toCharArray
    private	var	glyphVec: GlyphVector = null
    private val glyphColor = new Color(0, 204, 204)
    private val frc: FontRenderContext = new FontRenderContext(null, true, true)
    
    private var glyphWidth    = 0
    private var glyphHeight   = 0
    private var glyphBaseline = 0
    private var xPosGlyph     = 0
    private var yPosGlyph     = 0
    private var lastXPosGlyph = 0
    private var lastYPosGlyph = 0
    private var xPosRange	  = 0
    private var yPosRange	  = 0
    private var isFirstPos    = true

    private var offsetS  	  = 0f // StrikethroughOffset < 0, =? middle of pixel height
    private var offsetU  	  = 0f // UnderlineOffset > 0, =? bottom pixel 
    private var drawXPosGlyph = 0
    private var drawYPosGlyph = 0
    private var xMin 	      = 0
    private var yMin 		  = 0
    private var xLength 	  = 0
    private var yLength 	  = 0
    private val bnds		  =	8 // increase the updated bg image area a little bit

    
    private[DistortStringUniverse] def imageSize(width: Int, height: Int) {
      newImgWidth = width
      newImgHeight = height
      postId(REZISEIMAGE)
    }
        
    private[DistortStringUniverse] def startStop(start: Boolean) {
      if (start) postId(START)
      else postId(STOP)
    }

    // speed = [1, 100]
    private[DistortStringUniverse] def speed(s: Int) {
        	
      // Loop duration determines rotation speed
      // New IncreasingAlphaDuration
      newDurationX = ( MinDurationX * 100f/s ).asInstanceOf[Long]
      newDurationY = ( MinDurationY * 100f/s ).asInstanceOf[Long]
        	           
      // If is running
      if (!xAlpha.isPaused) { 
        val pauseTime = System.currentTimeMillis
        xAlpha.pause(pauseTime)
        yAlpha.pause(pauseTime)
        resumeAlphas(newDurationX, newDurationY)
      }
    }      
    
    // (Hint: This approach doesn't work for mode = Alpha.INCREASING_ENABLE | Alpha.DECREASING_ENABLE)
    // Resume Alphas with new duration/speed
    private def resumeAlphas(durationX: Long, durationY: Long) {
    
      val oldStartTimeX = xAlpha.getStartTime
      val oldStartTimeY = yAlpha.getStartTime

      val pauseTimeXY = xAlpha.getPauseTime

      val pauseValueX = xAlpha.value
      val pauseValueY = yAlpha.value

      // Offset according to alpha's pauseValue and the new IncreasingAlphaDuration
      val resumeOffsetTimeX = (pauseValueX * durationX).asInstanceOf[Long]
      val resumeOffsetTimeY = (pauseValueY * durationY).asInstanceOf[Long]

      xAlpha.setIncreasingAlphaDuration(durationX)
      yAlpha.setIncreasingAlphaDuration(durationY)

      // Resume

      // Alpha source code: newStartTime = oldStartTime + resumeTime - pauseTime
      // Start immediately and adapt new duration:
      //   => System.currentTimeMillis - resumeOffsetTime = oldStartTime + resumeTime - pauseTime
      //   => resumeTime = System.currentTimeMillis - resumeOffsetTime - oldStartTime + pauseTime

      val currTime = System.currentTimeMillis

      xAlpha.resume(currTime - resumeOffsetTimeX - oldStartTimeX + pauseTimeXY)
      yAlpha.resume(currTime - resumeOffsetTimeY - oldStartTimeY + pauseTimeXY)           	
    }
 
    // Behavior
    
    override def initialize = wakeupOn(resizeStart)

    override def processStimulus(criteria: JEnumeration[_]) { // renamed java.util.Enumeration
      while (criteria.hasMoreElements) {    	  
        criteria.nextElement match {
        	
          // 2D string animation
          case w: WakeupOnElapsedFrames =>
            if (!xAlpha.isPaused) 
              animateString
            wakeupOn(frameResizeStop)
            
          // Resize, Start/Stop
          case w: WakeupOnBehaviorPost =>
            
            w.getTriggeringPostId match {
            	
              case REZISEIMAGE =>
                setupImage(newImgWidth, newImgHeight)    

                if (xAlpha.isPaused) {
                  animateString
                  wakeupOn(resizeStart)
                }
                else 
                  wakeupOn(frameResizeStop)
                    	
              case START =>
                if (xAlpha.getIncreasingAlphaDuration != newDurationX)
                  resumeAlphas(newDurationX, newDurationY)
                else {                    		
                  xAlpha.resume
                  yAlpha.resume
                }
                wakeupOn(frameResizeStop)
              
              case STOP =>
                xAlpha.pause
                yAlpha.pause
                wakeupOn(resizeStart)
                return
                
              case _ => wakeupOn(resizeStart)
            }
            
          case _ => wakeupOn(resizeStart)
        }        
      }
    }
    
    // Resize background image and glyphs
    private def setupImage(width: Int, height: Int) {
	  if (bgImage ne null) bgImage.flush
	
	  currImgWidth = width
      currImgHeight = height
    
	  bgImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB)   
      val g2d = bgImage.getGraphics.asInstanceOf[Graphics2D]
	  g2d.setColor(bgColor)
	  g2d.fillRect(0, 0, width, height)
	  g2d.dispose
	
	  bgImageComp = new ImageComponent2D(ImageComponent.FORMAT_RGB, bgImage, true, false)
	  bgImageComp.setCapability(ImageComponent.ALLOW_IMAGE_WRITE)
	
      bg.setImage(bgImageComp)

      val font = new Font("Lucida Sans", Font.PLAIN, (width/20.0f).asInstanceOf[Int])
	
      val metrics = font.getLineMetrics(chars, 0, chars.length-1, frc)		
      offsetS = metrics.getStrikethroughOffset
      offsetU = metrics.getUnderlineOffset
	
      glyphVec = font.createGlyphVector(frc, chars)
    
      val pixRect = glyphVec.getPixelBounds(frc, 10, 100)
      glyphWidth = pixRect.width  
      glyphHeight = pixRect.height
	
      xPosRange = currImgWidth - glyphWidth
      yPosRange = currImgHeight - glyphHeight
	
      // baseline = glyphHeight/2-offsetS-offsetU => top char pixel will be drawn at panel border
      glyphBaseline = (glyphHeight/2-offsetS-offsetU + 0.5f).asInstanceOf[Int]
	
      isFirstPos = true
    }    
    
    // Called per frame
    private def animateString {
	
      if (!isFirstPos) {
        lastXPosGlyph = xPosGlyph
        lastYPosGlyph = yPosGlyph
      }
	
      // alpha value : [0, 1] -> [0, 1, 2] -> [0, 1, 0]
      var xValue = xAlpha.value*2
      var yValue = yAlpha.value*2
	
      if (xValue > 1) xValue = 2-xValue
      if (yValue > 1) yValue = 2-yValue

      xPosGlyph = (xPosRange * xValue).asInstanceOf[Int] //  + 0.5f
      yPosGlyph = (yPosRange * yValue + 0.5f).asInstanceOf[Int] 

      drawXPosGlyph = xPosGlyph - 2 //  - distance between left logical and pixel bound
      drawYPosGlyph = glyphBaseline + yPosGlyph
	
      if (isFirstPos) {
        isFirstPos = false
        lastXPosGlyph = xPosGlyph
        lastYPosGlyph = yPosGlyph
      }
	
      // Minimize update area = union of last and curr string's rectangle
	
      // x, width
      if (lastXPosGlyph < xPosGlyph) {
        xMin = lastXPosGlyph
        xLength = xPosGlyph-xMin + glyphWidth
      }
      else {
        xMin = xPosGlyph
        xLength = lastXPosGlyph-xMin + glyphWidth
      }
      if (xMin < bnds)
        xMin = 0
      else
        xMin -= bnds
      xLength += 2*bnds
      if (xMin + xLength > currImgWidth)
        xLength = currImgWidth - xMin
	
      // y, height
      if (lastYPosGlyph < yPosGlyph) {
        yMin = lastYPosGlyph
        yLength = yPosGlyph-yMin + glyphHeight
      }
      else {
        yMin = yPosGlyph
        yLength = lastYPosGlyph-yMin + glyphHeight
      }
      if (yMin < bnds)
        yMin = 0
      else
        yMin -= bnds
      yLength += 2*bnds
      if (yMin + yLength > currImgHeight)
        yLength = currImgHeight - yMin
	
      // Update background image
	  if (bgImageComp != null)
        bgImageComp.updateData(this, xMin, yMin, xLength, yLength)      	
    }    
    
    // Interface ImageComponent2D.Updater
    def updateData(imageComp: ImageComponent2D, x: Int, y: Int, width: Int, height: Int) {
      val g2d = bgImage.getGraphics.asInstanceOf[Graphics2D]

      g2d.setColor(bgColor)
      g2d.fillRect(x, y, width, height)
      g2d.setColor(glyphColor)
      // The baseline of the first character is at position (_, drawYPosGlyph) in the User Space
      g2d.drawGlyphVector(glyphVec, drawXPosGlyph, drawYPosGlyph)

      g2d.dispose       	
    }
  }
  
  // Distort 3D string 
    
  // Derived from "org.jdesktop.j3d.examples.distort_glyph.DistortBehavior.java", see license notice
  private object distortBehavior extends Behavior with GeometryUpdater {
   
	require(goldGeom ne null, "No goldGeom instance !!")
	require(woodAppearance ne null, "No woodAppearance instance !!")
	
    setSchedulingBounds(globalBounds)
    
    private val START = 98
    private val STOP  = 99
	  
    private val startCriterion = new WakeupOnBehaviorPost(this, START)
    private val stopFrameCriterion = new WakeupOr(
      Array(new WakeupOnBehaviorPost(this, STOP), new WakeupOnElapsedFrames(0))
    )
	  
    private val MinDuration = 10000L                                   // max speed
    private var newDuration = (MinDuration*100f/30).asInstanceOf[Long] // start speed

    private val alpha = new Alpha {
      setIncreasingAlphaDuration(newDuration)
      setLoopCount(-1)
      setStartTime(System.currentTimeMillis)
      pause
    }
    private var currAlphaValue = 0.0f
    private var lastAlphaValue = Float.NegativeInfinity
    
    private val TwoPi3 = Math.Pi*2*3
    private val TwoPi5 = Math.Pi*2*5
    
    
    private var spx = 0f
    private var spy = 0f
        
    private val coord = new Vector3f
    private val origCoordArray = goldGeom.getCoordRefFloat
    private val copyCoordArray = origCoordArray.clone.asInstanceOf[Array[Float]]
    private val coordsCt: Int = origCoordArray.length/3

    private val normal = new Vector3f
    private val origNormalArray = goldGeom.getNormalRefFloat
    private val copyNormalArray = origNormalArray.clone.asInstanceOf[Array[Float]]

    private val t3 = new Transform3D
    
    private[DistortStringUniverse] def startStop(start: Boolean) {
      if (start) postId(START)
      else postId(STOP)
    }
        
    // speed = [1, 100]
    private[DistortStringUniverse] def speed(s: Int) {
	           
      // Loop duration determines rotation speed
      // New IncreasingAlphaDuration
      newDuration = ( MinDuration * 100f/s ).asInstanceOf[Long]
    
      // is running
      if (!alpha.isPaused) { 
        alpha.pause(System.currentTimeMillis) // smoother
        resumeAlpha(newDuration)
      }
    }

    // Resume Alpha with new duration/speed
    private def resumeAlpha(duration: Long) {
	
      val oldStartTime = alpha.getStartTime
      val pauseTime = alpha.getPauseTime
      val pauseValue = alpha.value
	
      // Offset according to alpha's pauseValue and the new IncreasingAlphaDuration
      val resumeOffsetTime = (pauseValue * duration).asInstanceOf[Long]

      alpha.setIncreasingAlphaDuration(duration)

      // Resume

      // Alpha source code: newStartTime = oldStartTime + resumeTime - pauseTime
      // Start immediately and adapt new duration:
      //   => System.currentTimeMillis - resumeOffsetTime = oldStartTime + resumeTime - pauseTime
      //   => resumeTime = System.currentTimeMillis - resumeOffsetTime - oldStartTime + pauseTime

      alpha.resume(System.currentTimeMillis - resumeOffsetTime - oldStartTime + pauseTime)
    }    
    
    // Behavior
    
    override def initialize = wakeupOn(startCriterion)

    override def processStimulus(criteria: JEnumeration[_]) { // renamed java.util.Enumeration
      while (criteria.hasMoreElements) {   	  
        criteria.nextElement match {
        	
          // 3D string distortion
          case w: WakeupOnElapsedFrames =>
          
            currAlphaValue = alpha.value
            if (currAlphaValue != lastAlphaValue) {
              lastAlphaValue = currAlphaValue
                        
              spx = Math.sin(TwoPi3 * currAlphaValue).asInstanceOf[Float]
              spy = Math.cos(TwoPi5 * currAlphaValue).asInstanceOf[Float]
	
              if (isWoodShader)
                woodAppearance.attrValueSpXY(spx, spy)
              else
            	goldGeom.updateData(this)	                    
            }

            wakeupOn(stopFrameCriterion)
            
          // Start/Stop
          case w: WakeupOnBehaviorPost =>
            
            w.getTriggeringPostId match {
            	
              case START =>
                if (alpha.getIncreasingAlphaDuration != newDuration)
                  resumeAlpha(newDuration)
                else
                  alpha.resume
                    	
                wakeupOn(stopFrameCriterion)
              
              case STOP =>
                alpha.pause                   	
                // Synchronize vertices
                if (isWoodShader)
                  goldGeom.updateData(this)
                else 
                  woodAppearance.attrValueSpXY(spx, spy)
                        
                wakeupOn(startCriterion)
                return
              
              case _ => wakeupOn(startCriterion)
            }
            
          case _ => wakeupOn(startCriterion)
        }
      }
    }
        
    // Interface GeometryUpdater
    def updateData(geometry: Geometry) {
      t3.setIdentity
            
      var n = 0
      
      for (i <- 0 until coordsCt) {

        // Coordinates

        coord.x = copyCoordArray(n)
        coord.y = copyCoordArray(n+1)
        coord.z = copyCoordArray(n+2)

        val px = coord.x - spx
        val py = coord.y - spy
        val pz = coord.z

        val d = (Math.sqrt(px*px + py*py + pz*pz)).asInstanceOf[Float]

        //t3.rotZ(d)
        //t3.rotX(d*2)
        t3.rotY(d)
        t3.transform(coord)

        origCoordArray(n)   = coord.x
        origCoordArray(n+1) = coord.y
        origCoordArray(n+2) = coord.z

        // Normals

        normal.x = copyNormalArray(n)
        normal.y = copyNormalArray(n+1)
        normal.z = copyNormalArray(n+2)

        t3.transform(normal) // t3 is orthogonal

        origNormalArray(n)   = normal.x
        origNormalArray(n+1) = normal.y
        origNormalArray(n+2) = normal.z

        n += 3
      }
    }        
  }
  
  // Java 3D properties
  private[distortstring] def printJava3DProps {

    val vuProps = VirtualUniverse.getProperties

    println("Java 3D Version  =  " + vuProps.get("j3d.version"))
    println("Java 3D Renderer  =  " + vuProps.get("j3d.renderer"))
    println("Java 3D Pipeline  =  " + vuProps.get("j3d.pipeline"))
    println("------------------------------------------------------------------------")
  
    val c3dProps = view.getCanvas3D(0).queryProperties

    println("Native Version  =  " + c3dProps.get("native.version"))
    println("Native GLSL  =  " + c3dProps.get("shadingLanguageGLSL"))
    println("Native Vendor  =  " + c3dProps.get("native.vendor"))
    println("Native Renderer  =  " + c3dProps.get("native.renderer"))

    println("------------------------------------------------------------------------")
    println("")
  }
  
}
