package com.interactivemesh.j3d.testspace.jcanvas3d;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Toolkit;

import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import java.util.ArrayList;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JLabel;
import javax.swing.JLayeredPane;
import javax.swing.JPanel;
import javax.swing.JSlider;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

/**
 * TuxCubeGrayscalePanel.java
 *
 * Version: 1.0
 * Date: 2009/03/09
 *
 * Copyright (c) 2009
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * but you may not modify, compile, or distribute this 'TuxCubeGrayscalePanel.java'.
 *
 */
final class TuxCubeGrayscalePanel {

    private Font titleFont = null; 
    private Font textFont = null; 
    
    private Color bgColor = new Color(0.0f, 0.4f, 0.8f);        // blue
    private Color fgColor = new Color(1.0f, 1.0f, 1.0f);        // white
    
    private Color enterColor = new Color(1.0f, 0.7f, 0.0f);     // orange
    private Color pressColor = new Color(1.0f, 0.2f, 0.0f);     // dark orange            
    private Color selectColor = new Color(0.0f, 1.0f, 0.6f);    // neon green
    
    private int borderBLR = 50;
    private int borderT   = 80;
        
    private int space = 10;
    private int vspace = 5;

    
    private Label fpsValueLabel = null;
    
    private Slider rotationSlider = null;
    private Slider spotsizeSlider = null;
    
    private TuxCubeGrayscale universe = null;
    
    
    TuxCubeGrayscalePanel() {
        
    }
    
    // JApplet
    void destroy() {
        universe.closeUniverse();
    }
    
    void addTo(Container container, final JLayeredPane jLayeredPane) {
        
        System.out.println("TuxCubeGrayscale is running.");
        
        Dimension screenDim = Toolkit.getDefaultToolkit().getScreenSize();
        
        int screenHeight = screenDim.height;
        
        // screenHeight >= 1200
        int textFontSize = 18;
        int titleFontSize = 30;
        borderBLR = 50;
        borderT = 80;
        space = 10;

        // screenHeight  < 1024
        if (screenHeight < 1024) {
            textFontSize = 14;
            titleFontSize = 24;
            borderBLR = 30;
            borderT = 50;
            space = 5;
        }
        // 1024 <= screenHeight < 1200
        else if (screenHeight < 1200) {
            textFontSize = 16;
            titleFontSize = 26;
            borderBLR = 40;
            borderT = 60;
            space = 8;
        }

        titleFont = new Font("Dialog", Font.PLAIN, titleFontSize); 
        textFont = new Font("Dialog", Font.PLAIN, textFontSize); 

            
        universe = new TuxCubeGrayscale(this);
        
       
        // Parent of TuxCubeGrayscaleJC3DOB
        JPanel jCanvas3D = universe.getJCanvas3D();        
        
        //
        // Title
        //
        JLabel titleLabel = new JLabel();
        titleLabel.setFont(titleFont);
        titleLabel.setForeground(fgColor);
        titleLabel.setText("JCanvas3DOB Grayscale");
        titleLabel.setMinimumSize(new Dimension(100, borderT));
        titleLabel.setPreferredSize(new Dimension(100, borderT));
        titleLabel.setMaximumSize(new Dimension(Integer.MAX_VALUE, borderT));
        
        JPanel titlePanel = new JPanel();
        titlePanel.setLayout(new BoxLayout(titlePanel, BoxLayout.X_AXIS));
        titlePanel.setBackground(bgColor);
        titlePanel.setBorder(BorderFactory.createEmptyBorder(0, 2*space, 0, 0));
        
        titlePanel.add(titleLabel);
        titlePanel.add(Box.createHorizontalGlue());
        
        //
        // 3D Scene
        //
        JPanel scenePanel = new JPanel();
        scenePanel.setLayout(new BoxLayout(scenePanel, BoxLayout.X_AXIS));
        scenePanel.setBackground(bgColor);
        scenePanel.setBorder(BorderFactory.createLineBorder(Color.WHITE, 3));
        scenePanel.add(jCanvas3D, BorderLayout.CENTER);
        
        //
        // Actions
        //
        
        // VantagePoint
        
        JPanel vpPanel = new YPanel() {
            @Override
            public void paintComponent(Graphics g) {               
                super.paintComponent(g);                
                Dimension size = this.getSize();
                g.setColor(Color.WHITE);                
                // Horizontal line
                g.drawLine(0, size.height/2, size.width, size.height/2);
            }
        };        

        ButtonLabel frontButtom = new ButtonLabel("Front", 0, 5, 0, 5, "Front viewpoint");       
        ButtonLabel topButton = new ButtonLabel("Top", 0, 5, 0, 5, "Top viewpoint");
        
        ButtonLabelGroup vpGoup = new ButtonLabelGroup() {
            @Override
            void action(ButtonLabel button) {
                universe.setVantagePoint(button.getText());
            }           
        };
        vpGoup.add(frontButtom);
        vpGoup.add(topButton);
        
        vpPanel.add(frontButtom);
        vpPanel.add(Box.createVerticalStrut(vspace));
        vpPanel.add(topButton);
        
        // Projection
        
        JPanel projPanel = new YPanel() {
            @Override
            public void paintComponent(Graphics g) {               
                super.paintComponent(g);                
                Dimension size = this.getSize();
                g.setColor(Color.WHITE);                
                // Horizontal line
                g.drawLine(0, size.height/2, size.width, size.height/2);
            }
        };        
        
        RadioLabel paralRadio = new RadioLabel("Parallel", 0, 5, 0, 5, "Parallel projection mode");       
        RadioLabel perspRadio = new RadioLabel("Perspective", 0, 5, 0, 5, "Perspective projection mode");
        
        RadioLabelGroup projGroup = new RadioLabelGroup() {
            @Override
            void selectionChanged(RadioLabel radio) {
                universe.setProjectionMode(radio.getText());
            }
        };
        projGroup.add(paralRadio);
        projGroup.add(perspRadio);
        projGroup.setSelection(perspRadio);
        
        projPanel.add(paralRadio);
        projPanel.add(Box.createVerticalStrut(vspace));
        projPanel.add(perspRadio);
        
        // Cube dim
        
        JPanel cubePanel = new YPanel() {
            @Override
            public void paintComponent(Graphics g) {
                super.paintComponent(g);
                
                Dimension size = this.getSize();
                g.setColor(Color.WHITE);                
                // Horizontal line
                g.drawLine(0, size.height/2, size.width, size.height/2);
                // Vertical lines  
                int step = size.width/4;
                int b = (int)(size.height * 0.1);
                int yTop = b;
                int yBot = size.height - b;
                g.drawLine(  step, yTop,   step, yBot);
                g.drawLine(2*step, yTop, 2*step, yBot);
                g.drawLine(3*step, yTop, 3*step, yBot);                                
            }
        };
        
        String tip = " Tux per row / column";
        
        RadioLabel[] cubeRadios = new RadioLabel[8];
        
        for (int i=0; i < 8; i++) {
            String number = String.valueOf(i+1);
            cubeRadios[i] = new RadioLabel(number, 0, 5, 0, 5, number+tip);
        }
        
        RadioLabelGroup cubeGroup = new RadioLabelGroup() {
            @Override
            void selectionChanged(RadioLabel radio) {
                universe.setCubeDimension(Integer.parseInt(radio.getText()));
            }
        };
        
        for (int i=0; i < 8; i++)
            cubeGroup.add(cubeRadios[i]);
        
        cubeGroup.setSelection(cubeRadios[2]);
        
        JPanel topPanel = new XPanel();
        JPanel botPanel = new XPanel();
       
        for (int i=0; i < 4; i++)
            topPanel.add(cubeRadios[i]);
        for (int i=4; i < 8; i++)
            botPanel.add(cubeRadios[i]);
               
        cubePanel.add(topPanel);
        cubePanel.add(Box.createVerticalStrut(vspace));
        cubePanel.add(botPanel);
 
        
        // Frames per second
        
        JPanel fpsPanel = new YPanel();

        JLabel fpsLabel = new Label("F P S");
        fpsLabel.setForeground(Color.YELLOW);
        fpsLabel.setToolTipText("Frames per second");
        
        fpsValueLabel = new Label("0");
        fpsValueLabel.setForeground(Color.YELLOW);
        fpsValueLabel.setToolTipText("Current frames per second");

        fpsPanel.add(fpsLabel);
        fpsPanel.add(Box.createVerticalStrut(vspace));
        fpsPanel.add(fpsValueLabel);
        
        // Rotation
               
        JPanel rotationPanel = new YPanel();

        SingleButtonLabel rotationButtonLabel = new SingleButtonLabel(" <   Rotation   > ", "Reset rotation") {
            @Override
            void action() {
                universe.resetRotation();
                rotationSlider.setAtValue(50);
            }
        };
        
        Dimension rotLabelSize = rotationButtonLabel.getPreferredSize();
        Dimension sliderSize = new Dimension((int)(rotLabelSize.width*1.2), rotLabelSize.height);
        
        rotationSlider = new Slider(0, 100, 50, sliderSize) { 
            @Override
            void valueChanged(int value) {  
                universe.setRotationSpeed(value);            
            }
        };        
        
        rotationPanel.add(rotationButtonLabel);
        rotationPanel.add(Box.createVerticalGlue());
        rotationPanel.add(rotationSlider);
       
        // Navigation / Spot
        
        JPanel naviPanel = new YPanel() {
            @Override
            public void paintComponent(Graphics g) {               
                super.paintComponent(g);
                
                Dimension size = this.getSize();
                g.setColor(Color.WHITE);                
                // Horizontal line
                g.drawLine(0, size.height/2, size.width, size.height/2);
            }
        };        

        RadioLabel naviRadio = new RadioLabel("Navi", 0, 5, 0, 5, "Mouse navigation");       
        RadioLabel spotRadio = new RadioLabel("Spot", 0, 5, 0, 5, "Spot movement");
        
        RadioLabelGroup naviGroup = new RadioLabelGroup(){
            @Override
            void selectionChanged(RadioLabel radio) {
                universe.setInteractionMode(radio.getText());
            }
        };
        naviGroup.add(naviRadio);
        naviGroup.add(spotRadio);
        naviGroup.setSelection(naviRadio);
        
        naviPanel.add(naviRadio);
        naviPanel.add(Box.createVerticalStrut(vspace));
        naviPanel.add(spotRadio);
        
        // Spot size
        
        JPanel spotsizePanel = new YPanel();

        final int defaultSize = 22; // %
        
        SingleButtonLabel spotsizeButtonLabel = new SingleButtonLabel("Spot Size", "Reset spot size") {
            @Override
            void action() {
                spotsizeSlider.setValue(defaultSize);
            }
        };

        spotsizeSlider = new Slider(0, 100, defaultSize, sliderSize) { 
            @Override
            void valueChanged(int value) {  
                universe.setSpotSize(value);            
            }
        };        
        
        spotsizePanel.add(spotsizeButtonLabel);
        spotsizePanel.add(Box.createVerticalGlue());
        spotsizePanel.add(spotsizeSlider);
        
        //
        // Action panel
        //
        final JPanel actionPanel = new JPanel();
        actionPanel.setLayout(new BoxLayout(actionPanel, BoxLayout.X_AXIS));
        actionPanel.setOpaque(false);
        
        actionPanel.add(projPanel);
        actionPanel.add(Box.createHorizontalStrut(2*space));
        actionPanel.add(vpPanel);
        actionPanel.add(Box.createHorizontalStrut(2*space));
        actionPanel.add(cubePanel);
        actionPanel.add(Box.createHorizontalStrut(3*space));
        actionPanel.add(fpsPanel);
        actionPanel.add(Box.createHorizontalStrut(2*space));
        actionPanel.add(rotationPanel);
        actionPanel.add(Box.createHorizontalStrut(2*space));
        actionPanel.add(naviPanel);
        actionPanel.add(Box.createHorizontalStrut(2*space));
        actionPanel.add(spotsizePanel);
        
        Dimension actionSize = actionPanel.getPreferredSize();
        final int actionWidth = actionSize.width;
        final int actionHeight = actionSize.height;
                
        // Initial size
        int panelDim = (int)(Math.min(screenDim.width, screenDim.height) * 0.7f);       
        jCanvas3D.setPreferredSize(new Dimension(panelDim, panelDim));
        
        // Frame
        
        final JPanel framePanel = new JPanel();
        framePanel.setLayout(new BoxLayout(framePanel, BoxLayout.Y_AXIS));
        framePanel.setBackground(bgColor);
        framePanel.setBorder(BorderFactory.createEmptyBorder(0, borderBLR, borderBLR, borderBLR));
        
        framePanel.add(titlePanel);
        framePanel.add(scenePanel);
        
        // ActionPanel in PALETTE_LAYER
        
        framePanel.addComponentListener(new ComponentAdapter() { 
            public void componentResized(ComponentEvent event) {
                Dimension size = jLayeredPane.getSize();
                actionPanel.setBounds((size.width - actionWidth)/2, size.height-actionHeight-2*space-borderBLR, 
                                      actionWidth, actionHeight);     
                framePanel.revalidate(); 
                framePanel.repaint();
            }
        });
        
        jLayeredPane.add(actionPanel, JLayeredPane.PALETTE_LAYER);
               
        container.add(framePanel);

    }
    
    // Update GUI frames per second
    void setFPS(int fps) {
        fpsValueLabel.setText(Integer.toString(fps));
    }
    
    //
    // GUI components
    //

    final private class XPanel extends JPanel {
        private XPanel() {
            setLayout(new BoxLayout(this, BoxLayout.X_AXIS));
            setOpaque(false);
            setAlignmentY(JPanel.CENTER_ALIGNMENT);
        }
    }
    
    private class YPanel extends JPanel {
        private YPanel() {
            setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
            setOpaque(false);
            setAlignmentX(JPanel.CENTER_ALIGNMENT);
        }
    }
    
    private class Label extends JLabel {
        
        private boolean isPressed = false;
        private boolean isSelected = false;
        
        private Label(String text) {
            super(text);
            setFont(textFont);
            setForeground(fgColor);
            setAlignmentX(JPanel.CENTER_ALIGNMENT);
        }
        
        boolean isPressed() {
            return isPressed; 
        }
        void setPressed(boolean press) {
            isPressed = press;
        }
        
        boolean isSelected() {
            return isSelected; 
        }       
        void setSelected(boolean select) {
            isSelected = select;
        }
    }
    
    // Slider
    
    private class Slider extends JSlider {       
        
        private ChangeListener listener = null;
        
        private Slider(int min, int max, int value, Dimension preferredSize) {
            super(min, max, value);
            setOpaque(false);
            setAlignmentX(JPanel.CENTER_ALIGNMENT);
            setBorder(BorderFactory.createEmptyBorder());
            setPreferredSize(preferredSize);
            
            listener = new ChangeListener() {
                public void stateChanged(ChangeEvent event) {
                    valueChanged(Slider.this.getValue());
                }
            };            
            addChangeListener(listener);
        }
        
        private void setAtValue(int value) {
            this.removeChangeListener(listener);
            this.setValue(value);
            this.addChangeListener(listener);
        }
        
        void valueChanged(int value) {           
        }
    }
    
    // Button / RadioButton
    
    //
    // Button / -Group
    //
    // Used in a ButtonGroup
    private class ButtonLabel extends Label {
    
        private ButtonLabel(String text) {
            this(text, 0, 0, 0, 0, null);
        }
        private ButtonLabel(String text, String tip) {
            this(text, 0, 0, 0, 0, tip);
        }
        private ButtonLabel(String text, int t, int l, int b, int r, String tip) {
            super(text);
            setBorder(BorderFactory.createEmptyBorder(t, l, b, r));           
            if (tip != null)
                this.setToolTipText(tip);
        }
    }
    // Individual button
    private class SingleButtonLabel extends ButtonLabel {
        
        private SingleButtonLabel(String text) {
            this(text, 0, 0, 0, 0, null);
        }
        private SingleButtonLabel(String text, String tip) {
            this(text, 0, 0, 0, 0, tip);
        }
        private SingleButtonLabel(String text, int t, int l, int b, int r, String tip) {
            super(text, t, l, b, r, tip);
            
            addMouseListener(new MouseAdapter() { 

                public void mouseEntered(MouseEvent event) {
                    ButtonLabel button = (ButtonLabel)event.getSource();
                    if (!button.isPressed())
                        button.setForeground(enterColor);
                }
                public void mouseExited(MouseEvent event) {
                    ButtonLabel button = (ButtonLabel)event.getSource();
                    if (!button.isPressed())
                        button.setForeground(fgColor);
                }
                public void mousePressed(MouseEvent event) {
                    ButtonLabel button = (ButtonLabel)event.getSource();
                    button.setPressed(true);
                    button.setForeground(pressColor);
                }
                public void mouseReleased(MouseEvent event) {
                    ButtonLabel button = (ButtonLabel)event.getSource();
                    button.setPressed(false);
                    if (button.contains(event.getX(), event.getY())) {
                        button.setForeground(enterColor);                                  
                        action();
                    }
                    else
                        button.setForeground(fgColor);
                }
            });
        }
    
        void action() {           
        }
    }
    
    private class ButtonLabelGroup {
        
        // ButtonLabels of this group 
        private ArrayList<ButtonLabel> buttonList = null;
        
        // Single listener for all ButtonLabels
        private MouseAdapter actionListener = null;
        
        private ButtonLabelGroup() {
                       
            buttonList = new ArrayList<ButtonLabel>();
            
            actionListener = new MouseAdapter() { 

                public void mouseEntered(MouseEvent event) {
                    ButtonLabel button = (ButtonLabel)event.getSource();
                    if (!button.isPressed())
                        button.setForeground(enterColor);
                }
                public void mouseExited(MouseEvent event) {
                    ButtonLabel button = (ButtonLabel)event.getSource();
                    if (!button.isPressed())
                        button.setForeground(fgColor);
                }
                public void mousePressed(MouseEvent event) {
                    ButtonLabel button = (ButtonLabel)event.getSource();
                    button.setPressed(true);
                    button.setForeground(pressColor);
                }
                public void mouseReleased(MouseEvent event) {
                    ButtonLabel button = (ButtonLabel)event.getSource();
                    button.setPressed(false);
                    if (button.contains(event.getX(), event.getY())) {
                        button.setForeground(enterColor);                                  
                        action(button);
                    }
                    else
                        button.setForeground(fgColor);
                }
            };
        }
        
        // Use 'SingleButtonLabel' if an individual action is required as well
        private void add(ButtonLabel button) {
            buttonList.add(button);
            button.addMouseListener(actionListener);
        }
        
        void action(ButtonLabel button) {
        }
    }
    
    //
    // RadioButton / -Group
    //   
    final private class RadioLabel extends Label {
        
        private RadioLabel(String text) {
            this(text, 0, 0, 0, 0, null);
        }
        private RadioLabel(String text, int t, int l, int b, int r, String tip) {
            super(text);
            setBorder(BorderFactory.createEmptyBorder(t, l, b, r));    
            if (tip != null)
                this.setToolTipText(tip);
        }
    }
    
    private class RadioLabelGroup {
        
        // RadioLabels of this group 
        private ArrayList<RadioLabel> radioList = null;
        
        // Single listener for all RadioLabels
        private MouseAdapter actionListener = null;
        
        private RadioLabelGroup() {
                       
            radioList = new ArrayList<RadioLabel>();
            
            actionListener = new MouseAdapter() { 
                               
                @Override
                public void mouseEntered(MouseEvent event) {
                    RadioLabel radio = (RadioLabel)event.getSource();
                    if (!radio.isPressed() && !radio.isSelected())
                        radio.setForeground(enterColor);
                }
                @Override
                public void mouseExited(MouseEvent event) {
                    RadioLabel radio = (RadioLabel)event.getSource();
                    if (!radio.isPressed() && !radio.isSelected())
                        radio.setForeground(fgColor);
                }
                @Override
                public void mousePressed(MouseEvent event) {
                    RadioLabel radio = (RadioLabel)event.getSource();
                    if (!radio.isSelected()) {
                        radio.setPressed(true);
                        radio.setForeground(pressColor);
                    }
                }
                @Override
                public void mouseReleased(MouseEvent event) {
                    RadioLabel radio = (RadioLabel)event.getSource();
                    
                    if (radio.isSelected())
                        return;
                    
                    radio.setPressed(false);
                    if (radio.contains(event.getX(), event.getY())) {
                         select(radio);
                    }
                    else {
                        radio.setForeground(fgColor);
                    }
                }
            };
        }
        
        private void add(RadioLabel radio) {
            radioList.add(radio);
            radio.addMouseListener(actionListener);
        }
        
        private void setSelection(RadioLabel radio) {
            for (RadioLabel rl : radioList) {
                rl.setSelected(false);
                rl.setForeground(fgColor);
            }
            
            radio.setSelected(true);
            radio.setForeground(selectColor);
        }
        
        private void select(RadioLabel radio) {
            setSelection(radio);
            selectionChanged(radio);
        }
        
        void selectionChanged(RadioLabel radio) {
        }
    }   
}
